/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */

package sarj;

import javax.swing.*;
import javax.swing.table.*;
import java.util.*;

/**
 * The TimedScriptListTableModel class defines a model to use with the timed
 * script list table on the Script dialog
 */
class TimedScriptListTableModel extends AbstractTableModel
{
	private String[] columns = {"Script filename", "Type", "Enabled"};
	private Vector<Object[]> scriptData;
	
	private static final long serialVersionUID = 5947847346149275958L;
	
	/**
	 * Creates a new TimedScriptListTableModel object
	 */
	public TimedScriptListTableModel()
	{
		scriptData = new Vector<Object[]>();
	}
	
	/**
	 * Returns the number of rows currently in the table
	 *
	 * @return the number of rows currently in the table
	 */
	public int getRowCount()
	{
		return scriptData.size();
	}
	
	/**
	 * Returns the number of columns currently in the table
	 *
	 * @return the number of columns currently in the table
	 */
	public int getColumnCount()
	{
		return columns.length;
	}
	
	/**
	 * Returns the items in a row of the table
	 *
	 * @param row the 0-based index of the row to retrieve
	 *
	 * @return an array of objects containing the elements in each cell in the
	 *			specified row
	 */
	public Object []getRow(int row)
	{
		if(row < scriptData.size())
			return (Object [])scriptData.get(row);
		else return (Object[])null;
	}
	
	/**
	 * Returns an element in a cell of the table
	 *
	 * @param row the 0-based index of the row of the element to retrieve
	 * @param column the 0-based index of the column of the element to retrieve
	 *
	 * @return an object containing the element in the specified row and column
	 */
	public Object getValueAt(int row, int column)
	{
		if(row < scriptData.size())
		{
			Object []se = (Object [])scriptData.get(row);
			return se[column];
		}
		else return (Object[])null;
	}
	
	public void setValueAt(Object value, int row, int column)
	{
		if(row < scriptData.size() && column < 3)
		{
			Object []se = (Object [])scriptData.get(row);
			se[column] = value;
			fireTableCellUpdated(row,column);
		}
	}
	
	/**
	 * Returns the class of the object type of a column
	 *
	 * @param column the 0-based index of the column
	 *
	 * @return the class of objects in the specified column
	 */
	public Class getColumnClass(int column)
	{
		Object obj = getValueAt(0, column);
		if(obj != null)
        	return obj.getClass();
        else return (Class)null;
    }
    
    /**
     * Returns the name of a column
     *
     * @param column the 0-based index of the column
     *
     * @return a String containing the name of the column
     */
    public String getColumnName(int column)
    {
    	if(column < 3)
        	return columns[column];
        else return "Unknown";
    }
    
    /**
     * Clears all data held within the table
     */
    public void clear()
	{
		scriptData.clear();
		fireTableDataChanged();
	}
	
	/**
	 * Adds details of a script to the table
	 *
	 * @param name the name of the script event
	 * @param enabled the state of the event
	 * @param scriptFile the file path to execute when this event occurs
	 *
	 * @return the index in the script table of the newly added script
	 */
	public int addScript(String scriptFile, String type, boolean enabled, int interval, int time)
	{
		Object newEntry[] = new Object[5];
		newEntry[0] = scriptFile;
		newEntry[1] = type;
		newEntry[2] = new Boolean(enabled);
		newEntry[3] = new Integer(interval);
		newEntry[4] = new Integer(time);
		
		scriptData.add(newEntry);
		fireTableDataChanged();
		
		return scriptData.size() - 1;
	}
	
	/**
	 * Updates the state of a row within the table
	 *
	 * @param rowNum the row number in the table
	 * @param name the new name of the script event
	 * @param scriptFile the new file path to execute when this event occurs
	 * @param enabled the new state of the event
	 */
	public void updateScript(int rowNum, String scriptFile, String type, boolean enabled, int interval, int time)
	{
		Object entryData[] = scriptData.get(rowNum);
		if(scriptFile != null) entryData[0] = scriptFile;
		if(type != null) entryData[1] = type;
		entryData[2] = new Boolean(enabled);
		entryData[3] = new Integer(interval);
		entryData[4] = new Integer(time);
		
		fireTableDataChanged();
	}
	
	/**
	 * Removes a script from the table
	 *
	 * @param index the index of the item to delete
	 */
	public void removeScript(int index)
	{
		scriptData.remove(index);
		fireTableDataChanged();
	}
	
	/**
	 * Returns all the data within the table
	 *
	 * @return a 2D array of objects containing the data within the table fields
	 */
	 public Object[][] getData()
	 {
	 	int rowCount = getRowCount();
	 	int colCount = getColumnCount();
	 	Object [][]data = new Object[rowCount][colCount];
	 	
	 	if(rowCount == 0)
	 		return (Object[][])null;
	 	
	 	for(int i=0; i<rowCount; i++)
	 	{
	 		data[i] = getRow(i);
	 	}
	 	
	 	return data;
	 }
}