/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */

package sarj;

import javax.swing.*;
import javax.swing.table.*;
import java.util.*;

/**
 * The TeamTableModel class defines a model to use with the team table on the
 * main GUI window
 */
class TeamTableModel extends AbstractTableModel
{
	private String[] columns = {"Team", "Score", "Kills", "Deaths", "Ping"};
	private Vector<Object[]> teamData;
	
	private static final long serialVersionUID = 5947847346149275958L;
	
	/**
	 * Creates a new TeamTableModel object
	 */
	public TeamTableModel()
	{
		teamData = new Vector<Object[]>();
	}
	
	/**
	 * Returns the number of rows currently in the table
	 *
	 * @return the number of rows currently in the table
	 */
	public int getRowCount()
	{
		return teamData.size();
	}
	
	/**
	 * Returns the number of columns currently in the table
	 *
	 * @return the number of columns currently in the table
	 */
	public int getColumnCount()
	{
		return columns.length;
	}
	
	/**
	 * Returns all the data within the table
	 *
	 * @return a 2D array of objects containing the data within the table fields
	 */
	 public Object[][] getData()
	 {
	 	int rowCount = getRowCount();
	 	int colCount = getColumnCount();
	 	Object [][]data = new Object[rowCount][colCount];
	 	
	 	if(rowCount == 0)
	 		return new Object[][]{{}};
	 	
	 	for(int i=0; i<rowCount; i++)
	 	{
	 		data[i] = getRow(i);
	 	}
	 	
	 	return data;
	 }
	
	/**
	 * Returns the items in a row of the table
	 *
	 * @param row the 0-based index of the row to retrieve
	 *
	 * @return an array of objects containing the elements in each cell in the
	 *			specified row
	 */
	public Object []getRow(int row)
	{
		if(row < teamData.size())
		{
			Object tableData[] = (Object [])teamData.get(row);
			
			// clone the result (because we're overwriting some of the fields)
			Object result[] = tableData.clone();
			
			// Filter out the HTML tags
			if(((String)result[0]).contains("Alpha")) result[0] = "Alpha";
			else if(((String)result[0]).contains("Bravo")) result[0] = "Bravo";
			else if(((String)result[0]).contains("Charlie")) result[0] = "Charlie";
			else if(((String)result[0]).contains("Delta")) result[0] = "Delta";
			
			return result;
		}
		else return (Object[])null;
	}
	
	/**
	 * Returns an element in a cell of the table
	 *
	 * @param row the 0-based index of the row of the element to retrieve
	 * @param column the 0-based index of the column of the element to retrieve
	 *
	 * @return an object containing the element in the specified row and column
	 */
	public Object getValueAt(int row, int column)
	{
		if(row < teamData.size())
		{
			Object []te = (Object [])teamData.get(row);
			return te[column];
		}
		else return (Object[])null;
	}
	
	/**
	 * Returns the class of the object type of a column
	 *
	 * @param column the 0-based index of the column
	 *
	 * @return the class of objects in the specified column
	 */
	public Class getColumnClass(int c)
	{
		Object obj = getValueAt(0, c);
		if(obj != null)
        	return obj.getClass();
        else return (Class)null;
    }
    
    /**
     * Returns the name of a column
     *
     * @param column the 0-based index of the column
     *
     * @return a String containing the name of the column
     */
    public String getColumnName(int col)
    {
    	if(col < 5)
        	return columns[col];
        else return "Unknown";
    }
    
    /**
     * Clears all data held within the table
     */
    public void clear()
	{
		teamData.clear();
		fireTableDataChanged();
	}
	
	/**
	 * Adds details of a team to the table
	 *
	 * @param name the name of the team
	 * @param score the team's score
	 * @param kills the team's total number of kills from players
	 * @param deaths the team's total number of deaths from players
	 * @param ping the average ping of players on the team
	 */
	public void addTeam(String name, int score, int kills, int deaths, int ping)
	{
		Object newEntry[] = new Object[5];
		newEntry[0] = name;
		newEntry[1] = new Integer(score);
		newEntry[2] = new Integer(kills);
		newEntry[3] = new Integer(deaths);
		newEntry[4] = new Integer(ping);
		
		teamData.add(newEntry);
		fireTableDataChanged();
	}
}