/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */

package sarj;

/**
 * This interface provides a basic set of functions to make scripting easier. The
 * facade does not add any additional functionality, but simply 'wraps up' existing
 * functionality to make it more easily accessible by a script.
 */
public interface ScriptingFacade
{
	/**
	 * Returns the name of the map currently being played on the server
	 *
	 * @return a {@link String} containing the name of the current map
	 */
	public String getCurrentMap();
		
	/**
	 * Returns the type of game running on the server
	 *
	 * @return a {@link String} containing the name of the current game type
	 */
	public String getGameType();
		
	/**
	 * Returns the time limit of the current game
	 *
	 * @return the current time limit, in seconds
	 */
	public int getTimeLimit();
	
	/**
	 * Returns the score limit of the game
	 *
	 * @return the score limit
	 */
	public int getScoreLimit();
		
	/**
	 * Returns the time left within the current game
	 *
	 * @return the time remaining, in seconds
	 */
	public int getTimeLeft();
		
	/**
	 * Returns the number of players connected to the server
	 *
	 * @return the number of players currently connected
	 */
	public int getNumPlayers();
	
	/**
	 * Sets the main SARJ window title to that specified
	 *
	 * @param title the new title for the window
	 */
	public void setWindowTitle(String title);
	
	/**
	 * Returns the main SARJ window title
	 *
	 * @return a {@link String} containing the title of the main window
	 */
	 public String getWindowTitle();
	 
	 /**
	 * Returns the value of the IP field on the main window
	 *
	 * @return a {@link String} containing the value of the IP field
	 */
	public String getIPField();
	
	/**
	 * Returns the value of the port field on the main window
	 *
	 * @return a {@link String} containing the value of the port field
	 */
	public String getPortField();
	
	/**
	 * Returns the current value stored in the password field on the main window
	 *
	 * @return a String object containing the unmasked value of the password field
	 */
	public String getPasswordField();
	
	/**
	 * Returns the value of the command combo box on the main window
	 *
	 * @return a {@link String} containing the value of the command combo box
	 */
	public String getCommandBox();
	
	/**
	 * Sets the IP field on the main window
	 *
	 * @param ip the new IP address value
	 */
	public void setIPField(String ip);
	
	/**
	 * Sets the port field on the main window
	 *
	 * @param port the new port value
	 */
	public void setPortField(String port);
	
	/**
	 * Sets the password field on the main window
	 *
	 * @param password the new password value
	 */
	public void setPasswordField(String password);
	
	/**
	 * Sets the command combo box field on the main window
	 *
	 * @param command the new value to set the command combo box to
	 */
	public void setCommandBox(String command);
	
	/**
	 * Clears the console on the main window
	 */
	public void clearConsole();
	
	/**
	 * Disconnects the client from any connected server
	 */
	public void disconnectClient();
	
	/**
	 * Connects the client to the specified IP and port
	 *
	 * @param ip the IP address of the server
	 * @param port the port number of the server
	 */
	public void connectClient(String ip, int port);
	
	/**
	 * Connects the client to the IP and port displayed on the main window
	 */
	public void connectClient();
	
		/**
	 * Writes text to the main command console
	 *
	 * @param text the text to print on the console
	 */
	public void writeToConsole(String text);
	
	/**
	 * Sends data to a connected server
	 *
	 * @param data the data to send
	 */
	public void sendData(String data);
	
	/**
	 * Returns the row of the player table with the player name matching that
	 * provided
	 *
	 * @param playerName the name of the player to match
	 * @return the data contained in the table relating to the player
	 */
	public String[] getPlayerData(String playerName);
	
	/**
	 * Returns all data within the player table on the main window
	 *
	 * @return a 2D array of {@link String} objects containing the data in the
	 *		   table
	 */
	public String[][] getPlayerData();
	
	/**
	 * Returns the row of the team table with the team name matching that
	 * provided
	 *
	 * @param teamName the name of the team to match
	 * @return the data contained in the table relating to the team
	 */
	public String[] getTeamData(String teamName);
	
	/**
	 * Returns all data within the team table on the main window
	 *
	 * @return a 2D array of {@link String} objects containing the data in the
	 *		   table
	 */
	public String[][] getTeamData();
}