/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */
 
package sarj;

import javax.swing.*;
import javax.swing.table.*;
import javax.swing.event.*;
import java.awt.event.*;
import java.awt.*;
import java.io.*;

/**
 * Contains details of all time and event based scripts
 */
public class ScriptDialog extends JDialog implements ActionListener, FocusListener
{
	private static final long serialVersionUID = 5947847346149275958L;
	
	public final static String EXTENSION_js = "js";
    public final static String EXTENSION_es = "es";
    // Icon image locations
    private final static String openImageLocation = "/images/open.gif";
    private final static String addImageLocation = "/images/addsb.gif";
    private final static String deleteImageLocation = "/images/delete.gif";
    
	private String settingsFile;
	private GUI owner;
	
	// GUI components
	private JTabbedPane tabbedPane;
	
	private ImageIcon openImage, addImage, deleteImage;
	
	// Event driven scripts
	private JPanel eventScriptsPanel;
	private JTable eventScriptsTable;
	private JLabel eventScriptDetailsLabel;
	private JTextField eventScriptFilenameField;
	private JCheckBox eventScriptEnabledBox;
	private JButton eventScriptChooseButton;
	private JButton okButton, cancelButton;
	
	// Timed scripts
	private JPanel timedScriptsPanel;
	private JTable timedScriptsTable;
	private JRadioButton timedScriptDelayButton;
	private JTextField timedScriptHoursDelayField;
	private JTextField timedScriptMinutesDelayField;
	private JTextField timedScriptSecondsDelayField;
	private JRadioButton timedScriptTimeButton;
	private JTextField timedScriptHoursTimeField;
	private JTextField timedScriptMinutesTimeField;
	private JComboBox timedScriptAMPMBox;
	private JTextField timedScriptFilenameField;
	private JCheckBox timedScriptEnabledBox;
	private JButton timedScriptChooseButton, timedScriptAddButton, timedScriptRemoveButton; 
	
	/**
	 * Script dialog constructor
	 * @param owner the parent GUI of the dialog
	 * @param title the title of the dialog window
	 * @param settingsFile the file path of the script settings file to load
	 *					   from and save to
	 */
	public ScriptDialog(GUI owner, String title, String settingsFile)
	{
		super(owner.getMainFrame(), title, true);
		
		this.settingsFile = settingsFile;
		this.owner = owner;
		
		setLocationRelativeTo(null);
		getContentPane().setLayout(new BoxLayout(getContentPane(), BoxLayout.PAGE_AXIS));
		
		openImage = new ImageIcon(getClass().getResource(openImageLocation));
		addImage = new ImageIcon(getClass().getResource(addImageLocation));
		deleteImage = new ImageIcon(getClass().getResource(deleteImageLocation));
		
		tabbedPane = new JTabbedPane();
		
		// Event-driven scripts panel
		eventScriptsPanel = new JPanel();
		JPanel eventScriptsLeftPanel = new JPanel();
		JPanel eventScriptsCentrePanel = new JPanel();
		JPanel eventScriptsRightPanel = new JPanel();
		JPanel eventScriptDetailsPanel = new JPanel();
		JPanel bottomPanel = new JPanel();
		
		eventScriptDetailsLabel = new JLabel();
		eventScriptsLeftPanel.setLayout(new GridLayout(12, 1, 10, 13));
		eventScriptsCentrePanel.setLayout(new GridLayout(12, 1, 10, 9));
		eventScriptsRightPanel.setLayout(new GridLayout(12, 1, 10, 5));
		eventScriptsPanel.setLayout(new BoxLayout(eventScriptsPanel, BoxLayout.PAGE_AXIS));
		eventScriptsPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
		eventScriptDetailsPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5), BorderFactory.createTitledBorder("Script details")));
		eventScriptsTable = new JTable(new EventScriptListTableModel());
		eventScriptsTable.setToolTipText("Event script details");
		eventScriptsTable.setPreferredScrollableViewportSize(new Dimension(460, eventScriptsTable.getRowHeight() * 8));
		eventScriptsTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		eventScriptsTable.setForeground(Color.BLUE);
		eventScriptsTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		
		ListSelectionModel lsm = eventScriptsTable.getSelectionModel();
		lsm.addListSelectionListener(new ListSelectionListener()
		{
			public void valueChanged(ListSelectionEvent e)
			{
				if(e.getValueIsAdjusting())
				{
					// ignore extra messages
				}
			
				ListSelectionModel lsm = (ListSelectionModel)e.getSource();
	        	if (lsm.isSelectionEmpty())
	        	{
	        		// no row selected
	        	}
	        	else
	        	{
	            	int selectedRow = lsm.getMinSelectionIndex();
	            	EventScriptListTableModel tm = (EventScriptListTableModel)eventScriptsTable.getModel();
	            	Object []rowData = tm.getRow(selectedRow);
	            	loadEventScriptDetails((String)rowData[0], (String)rowData[1], ((Boolean)rowData[2]).booleanValue());
	        	}
			}
		});
		
		TableColumn tc;
		
		for(int i=0; i<3; i++)
		{
			tc = eventScriptsTable.getColumnModel().getColumn(i);
			switch(i)
			{
				case 0:	// Event name
					tc.setPreferredWidth(145);
					break;
				case 1:	// Script filename
					tc.setPreferredWidth(270);
					break;
				case 2:	// Enabled
					tc.setPreferredWidth(55);
					break;
				default:
					break;
			}
		}
		
		JScrollPane scriptScrollPane = new JScrollPane(eventScriptsTable);
		eventScriptFilenameField = new JTextField(20);
		eventScriptFilenameField.setToolTipText("This is the file path of the script to execute on the event");
		eventScriptFilenameField.addActionListener(this);
		eventScriptFilenameField.addKeyListener(new KeyAdapter()
		{
			public void keyReleased(KeyEvent e)
			{
				int pos = eventScriptFilenameField.getCaretPosition();
				updateCurrentEventSelection();
				eventScriptFilenameField.setCaretPosition(pos);
			}
		});
		eventScriptEnabledBox = new JCheckBox("Enable this event script");
		eventScriptEnabledBox.addActionListener(this);
		eventScriptChooseButton = new JButton(openImage);
		eventScriptChooseButton.setToolTipText("Click here to select a script file to associate with this event");
		eventScriptChooseButton.addActionListener(this);
		eventScriptChooseButton.setPreferredSize(new Dimension(25, 25));
		eventScriptChooseButton.setMinimumSize(new Dimension(25, 25));
		
		eventScriptFilenameField.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		setEventScriptLabelText("", "", "");
		
		// Initially, disable all details related components
		eventScriptFilenameField.setEnabled(false);
		eventScriptEnabledBox.setEnabled(false);
		eventScriptChooseButton.setEnabled(false);
		
		eventScriptDetailsPanel.setLayout(new BoxLayout(eventScriptDetailsPanel, BoxLayout.LINE_AXIS));
		JPanel eventScriptDetailsLeftSubPanel = new JPanel();
		JPanel eventScriptDetailsRightSubPanel = new JPanel();
		JPanel fileChoosePanel = new JPanel();
		eventScriptDetailsLeftSubPanel.setLayout(new BoxLayout(eventScriptDetailsLeftSubPanel, BoxLayout.PAGE_AXIS));
		eventScriptDetailsRightSubPanel.setLayout(new BoxLayout(eventScriptDetailsRightSubPanel, BoxLayout.PAGE_AXIS));
		
		eventScriptDetailsLeftSubPanel.setPreferredSize(new Dimension(200, 70));
		
		eventScriptDetailsLeftSubPanel.add(eventScriptDetailsLabel);
		
		fileChoosePanel.setLayout(new BoxLayout(fileChoosePanel, BoxLayout.LINE_AXIS));
		fileChoosePanel.setAlignmentX(Component.LEFT_ALIGNMENT);
		fileChoosePanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		fileChoosePanel.add(eventScriptFilenameField);
		fileChoosePanel.add(Box.createHorizontalStrut(10));
		fileChoosePanel.add(eventScriptChooseButton);
		
		eventScriptDetailsRightSubPanel.add(new JLabel("Script to execute:"));
		eventScriptDetailsRightSubPanel.add(fileChoosePanel);
		eventScriptDetailsRightSubPanel.add(eventScriptEnabledBox);
		
		eventScriptDetailsPanel.add(eventScriptDetailsLeftSubPanel);
		eventScriptDetailsPanel.add(eventScriptDetailsRightSubPanel);
		
		eventScriptsPanel.add(scriptScrollPane);
		eventScriptsPanel.add(eventScriptDetailsPanel);
		
		// Timed scripts panel
		
		timedScriptsTable = new JTable(new TimedScriptListTableModel());
		timedScriptsTable.setToolTipText("Timed script details");
		timedScriptsTable.setPreferredScrollableViewportSize(new Dimension(460, eventScriptsTable.getRowHeight() * 8));
		timedScriptsTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		timedScriptsTable.setForeground(Color.BLUE);
		timedScriptsTable.setSelectionMode(ListSelectionModel.SINGLE_SELECTION);
		
		lsm = timedScriptsTable.getSelectionModel();
		lsm.addListSelectionListener(new ListSelectionListener()
		{
			public void valueChanged(ListSelectionEvent e)
			{
				if(e.getValueIsAdjusting())
				{
					// ignore extra messages
				}
			
				ListSelectionModel lsm = (ListSelectionModel)e.getSource();
	        	if (lsm.isSelectionEmpty())
	        	{
	        		// no row selected
	        	}
	        	else
	        	{
	            	int selectedRow = lsm.getMinSelectionIndex();
	            	TimedScriptListTableModel tm = (TimedScriptListTableModel)timedScriptsTable.getModel();
	            	Object []rowData = tm.getRow(selectedRow);
	            	loadTimedScriptDetails((String)rowData[0], (String)rowData[1], ((Boolean)rowData[2]).booleanValue(), ((Integer)rowData[3]).intValue(), ((Integer)rowData[4]).intValue());
	        	}
			}
		});
		
		for(int i=0; i<2; i++)
		{
			tc = timedScriptsTable.getColumnModel().getColumn(i);
			switch(i)
			{
				case 0:	// Script filename
					tc.setPreferredWidth(350);
					break;
				case 1:	// Type
					tc.setPreferredWidth(60);
					break;
				case 2:	// Enabled
					tc.setPreferredWidth(60);
					break;
				default:
					break;
			}
		}
		
		// TEMP
		TimedScriptListTableModel model = (TimedScriptListTableModel)timedScriptsTable.getModel();
		model.addScript("test file 1", "Interval", true, 100, 100);
		model.addScript("test file 2", "Interval", false, 10, 10);
		model.addScript("test file 3", "Time", false, 10, 721);
		model.addScript("test file 4", "Interval", false, 10, 10);
		
		JScrollPane timedScriptsScrollPane = new JScrollPane(timedScriptsTable);
		
		// Add / Remove script buttons
		JPanel timedScriptAddRemovePanel = new JPanel();
		timedScriptAddRemovePanel.setLayout(new BoxLayout(timedScriptAddRemovePanel, BoxLayout.LINE_AXIS));
		timedScriptAddButton = new JButton("Add", addImage);
		timedScriptAddButton.setToolTipText("Add a new timed script");
		timedScriptAddButton.addActionListener(this);
		timedScriptRemoveButton = new JButton("Remove", deleteImage);
		timedScriptRemoveButton.setToolTipText("Remove the selected timed script");
		timedScriptRemoveButton.addActionListener(this);
		timedScriptAddRemovePanel.add(timedScriptAddButton);
		timedScriptAddRemovePanel.add(Box.createRigidArea(new Dimension(5, 0)));
		timedScriptAddRemovePanel.add(timedScriptRemoveButton);
		
		// Details of a timed script
		JPanel timedScriptDetailsPanel = new JPanel();
		timedScriptDetailsPanel.setLayout(new BoxLayout(timedScriptDetailsPanel, BoxLayout.PAGE_AXIS));
		timedScriptDetailsPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder("Script details"), BorderFactory.createEmptyBorder(5,5,5,5)));
		timedScriptDelayButton = new JRadioButton("Execute this script after a certain time delay");
		timedScriptDelayButton.addActionListener(this);
		timedScriptDetailsPanel.add(timedScriptDelayButton);
		timedScriptDetailsPanel.setAlignmentX(Component.CENTER_ALIGNMENT);
		
		// For the time interval  option - setting hours, minutes and seconds //
		JPanel intervalOptionSubPanel = new JPanel();
		intervalOptionSubPanel.setLayout(new BoxLayout(intervalOptionSubPanel, BoxLayout.LINE_AXIS));
		
		intervalOptionSubPanel.add(Box.createRigidArea(new Dimension(50,0)));
		intervalOptionSubPanel.add(new JLabel("Time interval: "));
		intervalOptionSubPanel.add(Box.createRigidArea(new Dimension(5,0)));
		timedScriptHoursDelayField = new JTextField(4);
		timedScriptHoursDelayField.setMaximumSize(new Dimension(30, 20));
		timedScriptHoursDelayField.setMinimumSize(new Dimension(30, 20));
		timedScriptHoursDelayField.setForeground(Color.BLUE);
		timedScriptHoursDelayField.setToolTipText("Number of hours delay between script execution");
		timedScriptHoursDelayField.addFocusListener(this);
		timedScriptHoursDelayField.addKeyListener(new KeyAdapter()
		{
			public void keyReleased(KeyEvent e)
			{
				updateCurrentTimedSelection();
			}
		});
		intervalOptionSubPanel.add(timedScriptHoursDelayField);
		intervalOptionSubPanel.add(Box.createRigidArea(new Dimension(5,0)));
		intervalOptionSubPanel.add(new JLabel("hours "));
		
		timedScriptMinutesDelayField = new JTextField(2);
		timedScriptMinutesDelayField.setMaximumSize(new Dimension(25, 20));
		timedScriptMinutesDelayField.setMinimumSize(new Dimension(25, 20));
		timedScriptMinutesDelayField.setForeground(Color.BLUE);
		timedScriptMinutesDelayField.setToolTipText("Number of minutes delay between script execution");
		timedScriptMinutesDelayField.addFocusListener(this);
		timedScriptMinutesDelayField.addKeyListener(new KeyAdapter()
		{
			public void keyReleased(KeyEvent e)
			{
				updateCurrentTimedSelection();
			}
		});
		intervalOptionSubPanel.add(timedScriptMinutesDelayField);
		intervalOptionSubPanel.add(Box.createRigidArea(new Dimension(5,0)));
		intervalOptionSubPanel.add(new JLabel("minutes "));
		
		intervalOptionSubPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
		timedScriptDelayButton.setAlignmentX(Component.LEFT_ALIGNMENT);
		
		timedScriptSecondsDelayField = new JTextField(2);
		timedScriptSecondsDelayField.setMaximumSize(new Dimension(25, 20));
		timedScriptSecondsDelayField.setMinimumSize(new Dimension(25, 20));
		timedScriptSecondsDelayField.setForeground(Color.BLUE);
		timedScriptSecondsDelayField.setToolTipText("Number of seconds delay between script execution");
		timedScriptSecondsDelayField.addFocusListener(this);
		timedScriptSecondsDelayField.addKeyListener(new KeyAdapter()
		{
			public void keyReleased(KeyEvent e)
			{
				updateCurrentTimedSelection();
			}
		});
		intervalOptionSubPanel.add(timedScriptSecondsDelayField);
		intervalOptionSubPanel.add(Box.createRigidArea(new Dimension(5,0)));
		intervalOptionSubPanel.add(new JLabel("seconds "));
		
		timedScriptDetailsPanel.add(intervalOptionSubPanel);
		////////////////////////////////////////////////////////////////////////
		
		timedScriptTimeButton = new JRadioButton("Execute this script at a specific time");
		timedScriptTimeButton.addActionListener(this);
		timedScriptDetailsPanel.add(timedScriptTimeButton);
		
		// For the specific time option - hour, minute and AM/PM ///////////////
		JPanel timeOptionSubPanel = new JPanel();
		timeOptionSubPanel.setLayout(new BoxLayout(timeOptionSubPanel, BoxLayout.LINE_AXIS));
		timeOptionSubPanel.setAlignmentX(Component.LEFT_ALIGNMENT);
		timeOptionSubPanel.add(Box.createRigidArea(new Dimension(50,0)));
		timeOptionSubPanel.add(new JLabel("Execution time: "));
		timeOptionSubPanel.add(Box.createRigidArea(new Dimension(5,0)));
		timedScriptHoursTimeField = new JTextField(2);
		timedScriptHoursTimeField.setMaximumSize(new Dimension(20, 20));
		timedScriptHoursTimeField.setMinimumSize(new Dimension(20, 20));
		timedScriptHoursTimeField.setForeground(Color.BLUE);
		timedScriptHoursTimeField.setToolTipText("The hour of the day to execute the script");
		timedScriptHoursTimeField.addFocusListener(this);
		timedScriptHoursTimeField.addKeyListener(new KeyAdapter()
		{
			public void keyReleased(KeyEvent e)
			{
				updateCurrentTimedSelection();
			}
		});
		timeOptionSubPanel.add(timedScriptHoursTimeField);
		timeOptionSubPanel.add(Box.createRigidArea(new Dimension(5,0)));
		timeOptionSubPanel.add(new JLabel(": "));
		timedScriptMinutesTimeField = new JTextField(2);
		timedScriptMinutesTimeField.setMaximumSize(new Dimension(20, 20));
		timedScriptMinutesTimeField.setMinimumSize(new Dimension(20, 20));
		timedScriptMinutesTimeField.setForeground(Color.BLUE);
		timedScriptMinutesTimeField.setToolTipText("Minutes passed the hour to execute the script");
		timedScriptMinutesTimeField.addFocusListener(this);
		timedScriptMinutesTimeField.addKeyListener(new KeyAdapter()
		{
			public void keyReleased(KeyEvent e)
			{
				updateCurrentTimedSelection();
			}
		});
		timeOptionSubPanel.add(timedScriptMinutesTimeField);
		timeOptionSubPanel.add(Box.createRigidArea(new Dimension(5,0)));
		timedScriptAMPMBox = new JComboBox(new String[]{"AM", "PM"});
		timedScriptAMPMBox.setMaximumSize(new Dimension(50, 20));
		timedScriptAMPMBox.setToolTipText("Select either AM or PM time");
		timedScriptAMPMBox.setForeground(Color.BLUE);
		timedScriptAMPMBox.addActionListener(this);
		timeOptionSubPanel.add(timedScriptAMPMBox);
		
		
		timedScriptDetailsPanel.add(timeOptionSubPanel);
		
		// Filename field and open file button /////////////////////////////////
		timedScriptFilenameField = new JTextField(20);
		timedScriptFilenameField.setToolTipText("This is the file path of the script to execute");
		timedScriptFilenameField.addActionListener(this);
		timedScriptFilenameField.addKeyListener(new KeyAdapter()
		{
			public void keyReleased(KeyEvent e)
			{
				int pos = timedScriptFilenameField.getCaretPosition();
				updateCurrentTimedSelection();
				timedScriptFilenameField.setCaretPosition(pos);
			}
		});
		
		timedScriptChooseButton = new JButton(openImage);
		timedScriptChooseButton.setToolTipText("Click here to select a script file");
		timedScriptChooseButton.addActionListener(this);
		timedScriptChooseButton.setPreferredSize(new Dimension(25, 25));
		timedScriptChooseButton.setMinimumSize(new Dimension(25, 25));
		
		timedScriptFilenameField.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		
		timedScriptEnabledBox = new JCheckBox("Enable this timed script");
		timedScriptEnabledBox.addActionListener(this);
		
		fileChoosePanel = new JPanel();
		fileChoosePanel.setLayout(new BoxLayout(fileChoosePanel, BoxLayout.LINE_AXIS));
		fileChoosePanel.setAlignmentX(Component.LEFT_ALIGNMENT);
		fileChoosePanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		fileChoosePanel.add(new JLabel("Script to execute: "));
		fileChoosePanel.add(timedScriptFilenameField);
		fileChoosePanel.add(Box.createHorizontalStrut(10));
		fileChoosePanel.add(timedScriptChooseButton);
		
		timedScriptDetailsPanel.add(fileChoosePanel);
		timedScriptDetailsPanel.add(timedScriptEnabledBox);
		
		ButtonGroup group = new ButtonGroup();
		group.add(timedScriptDelayButton);
		group.add(timedScriptTimeButton);
		
		////////////////////////////////////////////////////////////////////////
		
		// Whole panel for timed scripts to hold table & details
		JPanel timedScriptsPanel = new JPanel();
		timedScriptsPanel.setLayout(new BoxLayout(timedScriptsPanel, BoxLayout.PAGE_AXIS));
		timedScriptsPanel.setBorder(BorderFactory.createEmptyBorder(10, 10, 10, 10));
		timedScriptsPanel.add(timedScriptsScrollPane);
		timedScriptsPanel.add(Box.createRigidArea(new Dimension(0, 5)));
		timedScriptsPanel.add(timedScriptAddRemovePanel);
		timedScriptsPanel.add(timedScriptDetailsPanel);
		
		
		getContentPane().setLayout(new BorderLayout());
		
		//getContentPane().add(eventScriptsPanel, BorderLayout.CENTER);
		tabbedPane.addTab("Event scripts", eventScriptsPanel);
		tabbedPane.addTab("Timed scripts", timedScriptsPanel);
		getContentPane().add(tabbedPane, BorderLayout.CENTER);
		
		// Bottom panel - Ok & Cancel
		okButton = new JButton("Ok", Sarj.OK_ICON);
		okButton.setToolTipText("Save settings and close window");
		okButton.addActionListener(this);
		
		cancelButton = new JButton("Cancel", Sarj.CANCEL_ICON);
		cancelButton.setToolTipText("Cancel changes made");
		cancelButton.addActionListener(this);
		
		bottomPanel.add(okButton);
		bottomPanel.add(cancelButton);
		getContentPane().add(bottomPanel, BorderLayout.SOUTH);
		pack();
		
		setLocationRelativeTo(null);
		
		// Initially, grey out the script details
		timedScriptDelayButton.setEnabled(false);
		timedScriptHoursDelayField.setEnabled(false);
		timedScriptMinutesDelayField.setEnabled(false);
		timedScriptSecondsDelayField.setEnabled(false);
		timedScriptTimeButton.setEnabled(false);
		timedScriptHoursTimeField.setEnabled(false);
		timedScriptMinutesTimeField.setEnabled(false);
		timedScriptAMPMBox.setEnabled(false);
		timedScriptFilenameField.setEnabled(false);
		timedScriptChooseButton.setEnabled(false);
		timedScriptEnabledBox.setEnabled(false);
		timedScriptRemoveButton.setEnabled(false);
	}
	
	/**
	 * Adds details of an event script to the event script table
	 * @param name the name of the event related to the script
	 * @param filename the file path of the script to execute
	 * @param enabled whether or not this script should be enabled
	 */
	public void addEventScript(String name, String filename, boolean enabled)
	{
		EventScriptListTableModel sltm = (EventScriptListTableModel)eventScriptsTable.getModel();
		sltm.addScript(name, filename, enabled);
	}
	
	/**
	 * Loads the specified details of an item in the event script details table into
	 * the script information area below it, to provide more information
	 *
	 * @param name the name of the event related to the script
	 * @param filename the file path of the script to execute on this event
	 * @enabled whether or not this event script is enabled
	 */
	private void loadEventScriptDetails(String name, String filename, boolean enabled)
	{
		eventScriptChooseButton.setEnabled(true);
		eventScriptFilenameField.setEnabled(true);
		eventScriptEnabledBox.setEnabled(true);
		
		eventScriptEnabledBox.setSelected(enabled);
		eventScriptFilenameField.setText(filename);
		
		if("onLoad".equals(name))
		{
			setEventScriptLabelText(name, "When SARJ initially loads", "none");
			
		}
		else if("onExit".equals(name))
		{
			setEventScriptLabelText(name, "When user exits SARJ", "none");
			
		}
		else if("onConnect".equals(name))
		{
			setEventScriptLabelText(name, "When SARJ client establishes a connection with the server", "none");
			
		}
		else if("onDisconnect".equals(name))
		{
			setEventScriptLabelText(name, "When the connection to the server is lost", "none");
			
		}
		else if("onJoinRequest".equals(name))
		{
			setEventScriptLabelText(name, "When a player requests to join the server", "PLAYER_IP, PLAYER_PORT");
			
		}
		else if("onPlayerJoin".equals(name))
		{
			setEventScriptLabelText(name, "When a player joins the game", "PLAYER_NAME, PLAYER_IP, PLAYER_PORT");
		}
		else if("onPlayerLeave".equals(name))
		{
			setEventScriptLabelText(name, "When a player leaves the game", "PLAYER_NAME");	
		}
		else if("onAdminConnect".equals(name))
		{
			setEventScriptLabelText(name, "When an admin connects to the server", "none");
		}
		else if("onAdminDisconnect".equals(name))
		{
			setEventScriptLabelText(name, "When an admin disconnects from the server", "none");	
		}
		else if("onPlayerSpeak".equals(name))
		{
			setEventScriptLabelText(name, "When a player speaks", "PLAYER_NAME, SPEECH_TEXT");
		}
		else if("onTimeLeftNotify".equals(name))
		{
			setEventScriptLabelText(name, "When a 'time left' notification is sent from the server", "TIME_LEFT");
		}
		else if("onDataReceived".equals(name))
		{
			setEventScriptLabelText(name, "When data is received from the server", "DATA_STRING");
		}
		else
		{
			setEventScriptLabelText("Unknown", "Unknown event", "");
		}
	}
	
	/**
	 * Loads details of a timed script
	 *
	 * @param filename the file path of the script
	 * @param type the type of script ("Time" or "Interval")
	 * @param enabled whether or not this script is currently enabled
	 * @param interval the time interval, in seconds,  at which to execute the script
	 * @param time the time to execute the script, in minutes since 12:00 am
	 */
	private void loadTimedScriptDetails(String filename, String type, boolean enabled, int interval, int time)
	{
		timedScriptRemoveButton.setEnabled(true);
		timedScriptFilenameField.setEnabled(true);
		timedScriptFilenameField.setText(filename);
		timedScriptDelayButton.setEnabled(true);
		timedScriptTimeButton.setEnabled(true);
		timedScriptChooseButton.setEnabled(true);
		timedScriptEnabledBox.setEnabled(true);
		timedScriptEnabledBox.setSelected(enabled);	
		
		if("Interval".equals(type))
		{
			// Interval based
			timedScriptDelayButton.setSelected(true);
			timedScriptTimeButton.setSelected(false);
			timedScriptHoursDelayField.setEnabled(true);
			timedScriptMinutesDelayField.setEnabled(true);
			timedScriptSecondsDelayField.setEnabled(true);
			timedScriptHoursTimeField.setText("");
			timedScriptMinutesTimeField.setText("");
			timedScriptAMPMBox.setSelectedIndex(0);
			timedScriptHoursTimeField.setEnabled(false);
			timedScriptMinutesTimeField.setEnabled(false);
			timedScriptAMPMBox.setEnabled(false);
		}
		else
		{
			// Certain time based
			timedScriptDelayButton.setSelected(false);
			timedScriptTimeButton.setSelected(true);
			timedScriptHoursDelayField.setText("");
			timedScriptMinutesDelayField.setText("");
			timedScriptSecondsDelayField.setText("");
			timedScriptHoursDelayField.setEnabled(false);
			timedScriptMinutesDelayField.setEnabled(false);
			timedScriptSecondsDelayField.setEnabled(false);
			timedScriptHoursTimeField.setEnabled(true);
			timedScriptMinutesTimeField.setEnabled(true);
			timedScriptAMPMBox.setEnabled(true);
		}
			
		// Set the interval time
		int tempTime = interval;
		int hours = tempTime / 3600;	// how many hours in the time?
		tempTime %= 3600;
		int minutes = tempTime / 60;	// how many minutes?
		tempTime %= 60;					// seconds left here
			
		// Format the numbers to have a 0 prefix if needed
		
		timedScriptHoursDelayField.setText(Integer.toString(hours, 10));

		timedScriptMinutesDelayField.setText(Integer.toString(minutes, 10));	
		
		timedScriptSecondsDelayField.setText(Integer.toString(tempTime, 10));
		
		
		tempTime = time;
		if(tempTime >= 720)	// > 720 minutes = PM time
		{
			tempTime -=720;
			timedScriptAMPMBox.setSelectedIndex(1);
		}
		else
			timedScriptAMPMBox.setSelectedIndex(0);
			
		hours = tempTime / 60;
		tempTime %= 60;
			
		
		timedScriptHoursTimeField.setText(Integer.toString(hours, 10));
		
		timedScriptMinutesTimeField.setText(Integer.toString(tempTime, 10));
		
	}
	
	/**
	 * Called as a result of implementing ActionListener
	 *
	 * @param e an object containing details of the event
	 */
	public void actionPerformed(ActionEvent e)
	{
		Object source = e.getSource();
		
		if(source == eventScriptChooseButton)
		{
			JFileChooser fileChooser = new JFileChooser();
			fileChooser.setFileFilter(new javax.swing.filechooser.FileFilter()
			{
				public boolean accept(File f)
				{
				    if (f.isDirectory())
				    {
						return true;
				    }
				
				    String extension = getExtension(f);
				    if (extension != null)
				    {
						if (extension.equals(EXTENSION_js) ||
					    extension.equals(EXTENSION_es))
					    {
					        return true;
					    }
					    else
					    {
					    	return false;
						}
    				}
    				return false;
				}
				
				public String getDescription()
				{
					return "JavaScript / EcmaScript files";
				}
			});
			
			int value = fileChooser.showOpenDialog(this);
			
			if(value == JFileChooser.APPROVE_OPTION)
			{
				File theFile = fileChooser.getSelectedFile();
				String path = theFile.getAbsolutePath();
				
				eventScriptFilenameField.setText(path);
				updateCurrentEventSelection();
			}
		}
		else if(source == eventScriptEnabledBox)
		{
			updateCurrentEventSelection();
		}
		else if(source == eventScriptFilenameField)
		{
			updateCurrentEventSelection();
		}
		else if(source == timedScriptDelayButton)
		{
			timedScriptHoursTimeField.setEnabled(false);
			timedScriptMinutesTimeField.setEnabled(false);
			timedScriptAMPMBox.setEnabled(false);
			timedScriptHoursDelayField.setEnabled(true);
			timedScriptMinutesDelayField.setEnabled(true);
			timedScriptSecondsDelayField.setEnabled(true);
			
			updateCurrentTimedSelection();
		}
		else if(source == timedScriptTimeButton)
		{
			timedScriptHoursTimeField.setEnabled(true);
			timedScriptMinutesTimeField.setEnabled(true);
			timedScriptAMPMBox.setEnabled(true);
			timedScriptHoursDelayField.setEnabled(false);
			timedScriptMinutesDelayField.setEnabled(false);
			timedScriptSecondsDelayField.setEnabled(false);
			
			updateCurrentTimedSelection();
		}
		else if(source == timedScriptEnabledBox)
		{
			updateCurrentTimedSelection();
		}
		if(source == timedScriptChooseButton)
		{
			JFileChooser fileChooser = new JFileChooser();
			fileChooser.setFileFilter(new javax.swing.filechooser.FileFilter()
			{
				public boolean accept(File f)
				{
				    if (f.isDirectory())
				    {
						return true;
				    }
				
				    String extension = getExtension(f);
				    if (extension != null)
				    {
						if (extension.equals(EXTENSION_js) ||
					    extension.equals(EXTENSION_es))
					    {
					        return true;
					    }
					    else
					    {
					    	return false;
						}
    				}
    				return false;
				}
				
				public String getDescription()
				{
					return "JavaScript / EcmaScript files";
				}
			});
			
			int value = fileChooser.showOpenDialog(this);
			
			if(value == JFileChooser.APPROVE_OPTION)
			{
				File theFile = fileChooser.getSelectedFile();
				String path = theFile.getAbsolutePath();
				
				timedScriptFilenameField.setText(path);
				updateCurrentTimedSelection();
			}
		}
		else if(source == timedScriptAddButton)
		{
			TimedScriptListTableModel model = (TimedScriptListTableModel)timedScriptsTable.getModel();
			
			model.addScript("New script", "Interval", true, 0, 0);
		}
		else if(source == timedScriptRemoveButton)
		{
			TimedScriptListTableModel model = (TimedScriptListTableModel)timedScriptsTable.getModel();	
			ListSelectionModel lsm = timedScriptsTable.getSelectionModel();
    		int rowNum = lsm.getMinSelectionIndex();
			
			if(rowNum >= 0)
			{
				model.removeScript(rowNum);
				
				// Since there is now no valid selection, grey out some of the
				// components
				timedScriptDelayButton.setEnabled(false);
				timedScriptHoursDelayField.setEnabled(false);
				timedScriptMinutesDelayField.setEnabled(false);
				timedScriptSecondsDelayField.setEnabled(false);
				timedScriptTimeButton.setEnabled(false);
				timedScriptHoursTimeField.setEnabled(false);
				timedScriptMinutesTimeField.setEnabled(false);
				timedScriptAMPMBox.setEnabled(false);
				timedScriptFilenameField.setEnabled(false);
				timedScriptChooseButton.setEnabled(false);
				timedScriptEnabledBox.setEnabled(false);
				timedScriptRemoveButton.setEnabled(false);
			}
		}
		else if(source == cancelButton)
		{
			loadSettings();	// reload last saved settings to overwrite any changes
			setVisible(false);
		}
		else if(source == okButton)
		{
			saveSettings();
			setVisible(false);
		}
	}
	
	/**
	 * Called as a result of implementing FocusListener
	 *
	 * @param e an object containing details of the event
	 */
	public void focusGained(FocusEvent e)
	{
		//
	}
	
	/**
	 * Called as a result of implementing FocusListener
	 *
	 * @param e an object containing details of the event
	 */
	public void focusLost(FocusEvent e)
	{
		Object source = e.getSource();
		
		if(source == timedScriptHoursDelayField)
		{
			try
			{
				int value = Integer.parseInt(timedScriptHoursDelayField.getText());
				if(value < 0)
					timedScriptHoursDelayField.setText("00");
			}
			catch(NumberFormatException ex)
			{
				timedScriptHoursDelayField.setText("00");
			}
		}
		else if(source == timedScriptMinutesDelayField)
		{
			try
			{
				int value = Integer.parseInt(timedScriptMinutesDelayField.getText());
				if(value < 0)
					timedScriptMinutesDelayField.setText("00");
				else if(value > 59)
					timedScriptMinutesDelayField.setText("59");
			}
			catch(NumberFormatException ex)
			{
				timedScriptMinutesDelayField.setText("00");
			}
		}
		else if(source == timedScriptSecondsDelayField)
		{
			try
			{
				int value = Integer.parseInt(timedScriptSecondsDelayField.getText());
				if(value < 0)
					timedScriptSecondsDelayField.setText("00");
				else if(value > 59)
					timedScriptSecondsDelayField.setText("59");
			}
			catch(NumberFormatException ex)
			{
				timedScriptSecondsDelayField.setText("00");
			}
		}
		else if(source == timedScriptHoursTimeField)
		{
			try
			{
				int value = Integer.parseInt(timedScriptHoursTimeField.getText());
				if(value < 0)
					timedScriptHoursTimeField.setText("00");
				else if(value > 12)
					timedScriptHoursTimeField.setText("12");
			}
			catch(NumberFormatException ex)
			{
				timedScriptHoursTimeField.setText("00");
			}
		}
		else if(source == timedScriptMinutesTimeField)
		{
			try
			{
				int value = Integer.parseInt(timedScriptMinutesTimeField.getText());
				if(value < 0)
					timedScriptMinutesTimeField.setText("00");
				else if(value > 59)
					timedScriptMinutesTimeField.setText("59");
			}
			catch(NumberFormatException ex)
			{
				timedScriptMinutesTimeField.setText("00");
			}
		}
		
		updateCurrentTimedSelection();
	}
	
	/**
	 * Sets the text of the script details label
	 *
	 * @param name the name of the event related to the script
	 * @param desc a description of the event
	 * @param params a list of the parameters sent to a script dealing with
	 *				 this event
	 */
	private void setEventScriptLabelText(String name, String desc, String params)
	{
		eventScriptDetailsLabel.setText("<html><font size=4 color=#0000CC>" + name + "</font><br><font size=2 color=#000000></font><font size = #0000CC>Called: </font><font size=2 color=#000000>" + desc + "<br></font><font size = #0000CC>Parameters: </font><font size=2 color=#000000>" + params + "</font></html>");
	}

    /*
     * Get the extension of a file.
     *
     * @param f the file to get the extension of
     *
     * @return the file extension (excluding the '.')
     */
    private String getExtension(File f) {
        String ext = null;
        String s = f.getName();
        int i = s.lastIndexOf('.');

        if (i > 0 &&  i < s.length() - 1) {
            ext = s.substring(i+1).toLowerCase();
        }
        return ext;
    }
    
    /**
     * Updates the currently selected row in the event scripts table with the
     * information in the script details
     */
    private void updateCurrentEventSelection()
    {
    	int rowNum = eventScriptsTable.getSelectedRow();
    	
    	if(rowNum == -1)
    	{
    		Sarj.printError("Event script details update requested on current item when no item selected");
    		return;
    	}
    	
    	EventScriptListTableModel model = (EventScriptListTableModel)eventScriptsTable.getModel();
    	model.updateScript(rowNum, null, eventScriptFilenameField.getText(), eventScriptEnabledBox.isSelected());
    
    	ListSelectionModel lsm = eventScriptsTable.getSelectionModel();
    	lsm.setSelectionInterval(rowNum, rowNum);
    }
    
    /**
     * Updates the currently selected row in the timed scripts table with the
     * information in the script details
     */
     private void updateCurrentTimedSelection()
     {
     	int rowNum = timedScriptsTable.getSelectedRow();
     	
     	if(rowNum == -1)
     	{
     		Sarj.printError("Timed script details update requested on current item when no item selected");
     		return;
     	}
     	
     	TimedScriptListTableModel model = (TimedScriptListTableModel)timedScriptsTable.getModel();
     	// int rowNum, String scriptFile, String type, boolean enabled, int interval, int time
     	
     	String type = (timedScriptDelayButton.isSelected() ? "Interval" : "Time");
     	
     	// Calculate the valid time interval
     	int interval, intHours, intMins, intSecs;
     	try {intHours = Integer.parseInt(timedScriptHoursDelayField.getText()); } catch(NumberFormatException e) { intHours = 0; }
     	try {intMins = Integer.parseInt(timedScriptMinutesDelayField.getText()); } catch(NumberFormatException e) { intMins = 0; }
     	try {intSecs = Integer.parseInt(timedScriptSecondsDelayField.getText()); } catch(NumberFormatException e) { intSecs = 0; }
     	
	    interval =  + intHours * 3600 + intMins * 60 + intSecs;
     	
     	// Calculate the valid time
     	int time, timeHours, timeMins;
     	
     	try {timeHours = Integer.parseInt(timedScriptHoursTimeField.getText()); } catch(NumberFormatException e) { timeHours = 0; }
     	try {timeMins = Integer.parseInt(timedScriptMinutesTimeField.getText()); } catch(NumberFormatException e) { timeMins = 0; }
     	
     	if(timeHours > 23)
     		timeHours = 23;
     	if(timeHours > 12)
     	{
     		 timeHours -=12;
     		 timedScriptAMPMBox.setSelectedIndex(1);
     	}
     	
     	time = timeHours * 60 + timeMins;
     	
     	if(timedScriptAMPMBox.getSelectedIndex() == 1)
     	{
     		time += 720;
     	}
     	
     	model.updateScript(rowNum, timedScriptFilenameField.getText(), type, timedScriptEnabledBox.isSelected(), interval, time);
     	
     	ListSelectionModel lsm = timedScriptsTable.getSelectionModel();
    	lsm.setSelectionInterval(rowNum, rowNum);
     }
    
    /**
     * Loads the script dialog settings from the settings file
     */
    public void loadSettings()
	{
		BufferedReader br;
		String settingsData = "";
		String []tokens;
		
		try
		{
			br = new BufferedReader(new FileReader(new File(settingsFile)));
		}
		catch(Exception e)
		{
			JOptionPane.showMessageDialog(owner.getMainFrame(), "Failed to open script settings file!", "Error", JOptionPane.ERROR_MESSAGE);
			return;
		}
		try
		{
			settingsData = br.readLine();
		}
		catch(IOException ioe)
		{
			JOptionPane.showMessageDialog(owner.getMainFrame(), "Warning: Exception occurred when reading script settings file!", "Error", JOptionPane.ERROR_MESSAGE);
			return;
		}
		
		EventScriptListTableModel eventModel = (EventScriptListTableModel)eventScriptsTable.getModel();
		eventModel.clear();	// Clear the scripts table before loading in new data
		
		TimedScriptListTableModel timedModel = (TimedScriptListTableModel)timedScriptsTable.getModel();
		timedModel.clear();
				
		while(settingsData != null)
		{
			try
			{	
				String element;
				String []properties;
				
				if(settingsData != null && settingsData.length() > 0 && !settingsData.trim().startsWith("#"))
				{
					tokens = settingsData.split("=");
					if(tokens.length > 1)
					{
						// Remove padding spaces
						for(int i=0; i<tokens.length; i++)
							tokens[i] = tokens[i].trim();
						
						// Split tokens into element and properties
						element = tokens[0];
						properties = tokens[1].split(";");
						
						// Remove padding spaces from properties
						for(int i=0; i<properties.length; i++)
							properties[i] = properties[i].trim();
						
						// Check which settings element was set
						if("onLoad".equalsIgnoreCase(element) ||
							"onExit".equalsIgnoreCase(element) ||
							"onConnect".equalsIgnoreCase(element) ||
							"onDisconnect".equalsIgnoreCase(element) ||
							"onJoinRequest".equalsIgnoreCase(element) ||
							"onPlayerJoin".equalsIgnoreCase(element) ||
							"onPlayerLeave".equalsIgnoreCase(element) ||
							"onAdminConnect".equalsIgnoreCase(element) ||
							"onAdminDisconnect".equalsIgnoreCase(element) ||
							"onPlayerSpeak".equalsIgnoreCase(element) ||
							"onTimeLeftNotify".equalsIgnoreCase(element) ||
							"onDataReceived".equalsIgnoreCase(element))
						{
							if(properties.length < 2)
							{
								Sarj.printError("Incorrent number of properties in setting '" + settingsData + "'");
							}
							else
							{
								eventModel.addScript(element, properties[0], isTrue(properties[1]));
							}
						}
						else if("timed".equalsIgnoreCase(element))
						{
							if(properties.length < 5)
							{
								Sarj.printError("Incorrect number of properties in setting '" + settingsData + "'");
							}
							else
							{
								timedModel.addScript(properties[0], properties[1], isTrue(properties[2]), Integer.parseInt(properties[3]), Integer.parseInt(properties[4]));
							}
						}
					}
				}
				
				// Read and parse a line
				settingsData = br.readLine();
			}
			catch(IOException ioee)
			{
				JOptionPane.showMessageDialog(owner.getMainFrame(), "Warning: Failed to read from config file!", "Error", JOptionPane.ERROR_MESSAGE);
				return;
			}
		}
		try
		{
			br.close();
		}
		catch(IOException closeEx)
		{
			JOptionPane.showMessageDialog(owner.getMainFrame(), "Warning: Exception occurred when closing config file!", "Error", JOptionPane.ERROR_MESSAGE);
		}
	}
    
    /**
     * Saves the script settings to the settings file
     */
     public void saveSettings()
	 {
		BufferedWriter bw;
		int rows;
		String enabledString;
		
		EventScriptListTableModel eventModel = (EventScriptListTableModel)eventScriptsTable.getModel();
		TimedScriptListTableModel timedModel = (TimedScriptListTableModel)timedScriptsTable.getModel();
		
		rows = eventModel.getRowCount();
		try
		{
			bw = new BufferedWriter(new FileWriter(new File(settingsFile)));
			bw.write("# SARJ script settings\r\n\r\n# Event-driven scripts\r\n");
			for(int i=0; i<rows; i++)
			{
				enabledString = (((Boolean)eventModel.getValueAt(i, 2)).booleanValue() ? "true" : "false");
				bw.write((String)eventModel.getValueAt(i, 0) + "=" + (String)eventModel.getValueAt(i, 1) + ";" + enabledString + "\r\n");
			}
		
			rows = timedModel.getRowCount();
			
			bw.write("\r\n# Timed scripts\r\n");
			for(int i=0; i<rows; i++)
			{
				enabledString = (((Boolean)timedModel.getValueAt(i, 2)).booleanValue() ? "true" : "false");
				bw.write("timed=" + (String)timedModel.getValueAt(i, 0) + ";" + ((String)timedModel.getValueAt(i, 1)) + ";" + enabledString + ";" + ((Integer)timedModel.getValueAt(i, 3)) + ";" + ((Integer)timedModel.getValueAt(i, 4)) + "\r\n");
			}
			
			bw.close();
		}
		catch(IOException ioe)
		{
			JOptionPane.showMessageDialog(owner.getMainFrame(), "Warning: Failed to save script settings!", "Error", JOptionPane.ERROR_MESSAGE);
			return;
		}
	}
	
	/**
	 * A helper function which returns a boolean equivalent of the specified text
	 *
	 * @param text the text to evaluate as either 'true' or 'false'
	 * @return true if the text is interpreted as meaning 'true', or false otherwise
	 */
	private boolean isTrue(String text)
	{
		return ("1".equals(text) || "true".equalsIgnoreCase(text) || "yes".equalsIgnoreCase(text) || "on".equalsIgnoreCase(text));					
	}
	
	/**
	 * Returns the script file path associated with the specified event
	 *
	 * @param eventName the name of the event
	 * @return the filename associated with the event, or null if not found
	 * @throws EventNotFoundException if the specified event is not found
	 */
	public String getFilenameForEvent(String eventName) throws EventNotFoundException
	{
		EventScriptListTableModel model = (EventScriptListTableModel)eventScriptsTable.getModel();
		int numRows = model.getRowCount();
		String e;
		
		for(int i=0; i<numRows; i++)
		{
			e = (String)model.getValueAt(i, 0);
			if(eventName.equalsIgnoreCase(e))
				return (String)model.getValueAt(i, 1);;
		}
		
		throw new EventNotFoundException(eventName);
	}
	
	/**
	 * Returns whether or not the specified event is enabled
	 *
	 * @param eventName the name of the event
	 * @return whether of not the specified event is enabled.
	 * @throws EventNotFoundException if the specified event is not found
	 */
	public boolean isEventEnabled(String eventName) throws EventNotFoundException
	{
		EventScriptListTableModel model = (EventScriptListTableModel)eventScriptsTable.getModel();
		int numRows = model.getRowCount();
		String e;
		
		for(int i=0; i<numRows; i++)
		{
			e = (String)model.getValueAt(i, 0);
			if(eventName.equalsIgnoreCase(e))
				return ((Boolean)model.getValueAt(i, 2)).booleanValue();
		}
		
		throw new EventNotFoundException(eventName);
	}
	
	/**
	 * Overridden to disable the 'script details' related components
	 */
	public void setVisible(boolean b)
	{
		super.setVisible(b);
		
		if(b)
		{
			// Set the controls back to default values and set as disabled
			eventScriptFilenameField.setText("");
			eventScriptEnabledBox.setSelected(false);
			eventScriptFilenameField.setEnabled(false);
			eventScriptEnabledBox.setEnabled(false);
			eventScriptChooseButton.setEnabled(false);
			ListSelectionModel lsm = eventScriptsTable.getSelectionModel();
			lsm.clearSelection();
			
			timedScriptDelayButton.setEnabled(false);
			timedScriptHoursDelayField.setText("");
			timedScriptHoursDelayField.setEnabled(false);
			timedScriptMinutesDelayField.setText("");
			timedScriptMinutesDelayField.setEnabled(false);
			timedScriptSecondsDelayField.setText("");
			timedScriptSecondsDelayField.setEnabled(false);
			timedScriptTimeButton.setEnabled(false);
			timedScriptHoursTimeField.setText("");
			timedScriptHoursTimeField.setEnabled(false);
			timedScriptMinutesTimeField.setText("");
			timedScriptMinutesTimeField.setEnabled(false);
			timedScriptAMPMBox.setSelectedIndex(0);
			timedScriptAMPMBox.setEnabled(false);
			timedScriptFilenameField.setText("");
			timedScriptFilenameField.setEnabled(false);
			timedScriptChooseButton.setEnabled(false);
			timedScriptEnabledBox.setSelected(true);
			timedScriptEnabledBox.setEnabled(false);
			timedScriptRemoveButton.setEnabled(false);
			lsm = timedScriptsTable.getSelectionModel();
			lsm.clearSelection();
		}
	}
	
	/**
	 * Returns the details of all the interval based or specific time based scripts
	 *
	 * @return a 2d array of objects containing the details of the scripts
	 */
	 public Object[][]getTimedScriptDetails()
	 {
	 	TimedScriptListTableModel model = (TimedScriptListTableModel)timedScriptsTable.getModel();
	 	
	 	return model.getData();
	 }
}