/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */
 
package sarj;

import javax.swing.*;
import javax.swing.event.*;
import java.awt.*;
import java.awt.event.*;
import java.util.*;
import java.io.*;

/**
 * A custom swing component to hold all of the script buttons in one panel
 */
  
/*
  * Copyright 2005 Richard Davies
  *
  *
  * This file is part of SARJ.
  *
  * SARJ is free software; you can redistribute it and/or modify it under the terms of
  * the GNU General Public License as published by the Free Software Foundation;
  * either version 2 of the License, or (at your option) any later version.
  * 
  * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
  * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
  * PURPOSE. See the GNU General Public License for more details.
  *
  * You should have received a copy of the GNU General Public License along with
  * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
  * Suite 330, Boston, MA 02111-1307 USA 
*/

public class ScriptButtonPanel extends JPanel implements ActionListener, MouseListener
{
	private static final long serialVersionUID = 5947847346149275958L;
	
	private ArrayList<ScriptButton> buttonList;
	private GUIListener listener;
	private String settingsFile;
	private JPopupMenu sbMenu;
	private JMenuItem sbAddItem, sbEditItem, sbRemoveItem;
	private ScriptButton selectedButton = null;	// Need to keep track of this when
												// right-clicking over the button
	private EditScriptDialog editScriptDialog;
	
	/**
	 * Default constructor
	 */
	public ScriptButtonPanel(JFrame parent, String settingsFile, GUIListener listener)
	{
		super();
		this.settingsFile = settingsFile;
		this.listener = listener;
		buttonList = new ArrayList<ScriptButton>();
		
		sbMenu = new JPopupMenu("ScriptButton menu");
		sbAddItem = new JMenuItem("Add a new script button");
		sbAddItem.addActionListener(this);
		sbMenu.add(sbAddItem);
		sbEditItem = new JMenuItem("Edit this script button");
		sbEditItem.addActionListener(this);
		sbMenu.add(sbEditItem);
		sbRemoveItem = new JMenuItem("Remove this script button");
		sbRemoveItem.addActionListener(this);
		sbMenu.add(sbRemoveItem);
		
		setLayout(new FlowLayout(FlowLayout.LEFT, 5, 5));
		setToolTipText("Left click to launch, or right click to configure");
		
		loadSettings();
		
		editScriptDialog = new EditScriptDialog(parent, "Edit script button", this);
	}
	
	/**
	 * Adds a new ScriptButton to the panel
	 *
	 * @param name the name of the button. This will also appear as the button
	 *			   caption
	 * @param command the command string associated with the button, to be
	 *				  executed when it is activated
	 * @return the new script button
	 */
	public ScriptButton addScriptButton(String name, String command)
	{
		ScriptButton newSB = new ScriptButton(name, command);
		newSB.setToolTipText(command);
		newSB.addActionListener(this);
		newSB.addMouseListener(this);
		buttonList.add(newSB);
		add(newSB);
		revalidate();
		saveSettings();
		
		return newSB;
	}
	
	/**
	 * Removes a button from the script button panel
	 *
	 * @param name the name of the button to remove (the caption of the button)
	 */
	public void removeScriptButton(String name)
	{
		Iterator it = buttonList.iterator();
		ScriptButton sb;
		
		while(it.hasNext())
		{
			sb = (ScriptButton)it.next();
			
			if(name.equals(sb.getName()))
			{
				buttonList.remove(sb);
				remove(sb);
				revalidate();
				repaint();
				saveSettings();
				return;
			}
		}
	}
	
	/**
	 * Removes a button from the script button panel
	 *
	 * @param sb the ScripButton object to remove from the panel
	 */
	public void removeScriptButton(ScriptButton sb)
	{
		buttonList.remove(sb);
		remove(sb);
		revalidate();
		repaint();
		saveSettings();
	}
	
	/**
	 * Returns the buttons on the ScriptButton panel
	 *
	 * @return an array of the {@link ScriptButton} objects currently on the panel
	 */
	public ScriptButton[] getScriptButtons()
	{
		return (ScriptButton[])buttonList.toArray(new ScriptButton[buttonList.size()]);
	}
	
	/**
	 * Called as a result of implementing the ActionListener interface
	 *
	 * @param e an object containing details of the generated event
	 */
	public void actionPerformed(ActionEvent e)
	{
		Object source = e.getSource();
		
		if(source == sbAddItem)	// Add a new button to the panel
		{
			addNewButton();
		}
		else if(source == sbEditItem)	// Edit an existing button
		{
			editScriptDialog.associateWith(selectedButton);
			editScriptDialog.setVisible(true);
		}
		else if(source == sbRemoveItem)	// Remove a button from the panel
		{
			removeScriptButton(selectedButton);
		}
		else	// Can assume a ScriptButton was clicked
		{
			ScriptButton sb = (ScriptButton)e.getSource();
		
			listener.onScriptButtonClick(sb.getName(), sb.getScriptFile());
		}
	}
	
	/**
	 * Not implemented
	 */
	public void mouseExited(MouseEvent e)
	{
		//
	}
	
	/**
	 * Not implemented
	 */
	public void mouseEntered(MouseEvent e)
	{
		//
	}
	
	/**
	 * Tests to see if the popup menu needs to be shown
	 */
	public void mouseReleased(MouseEvent e)
	{
		maybeShowPopup(e);
	}
	
	/**
	 * Tests to see if the popup menu needs to be shown
	 */
	public void mousePressed(MouseEvent e)
	{
		maybeShowPopup(e);
	}
	
	/**
	 * Not implemented
	 */
	public void mouseClicked(MouseEvent e)
	{
		//
	}
	
	/**
	 * Helper function to show the popup menu if the event is relevant
	 *
	 * @param e the object containing details of the event
	 */
	private void maybeShowPopup(MouseEvent e)
	{
		if(e.isPopupTrigger() && e.getSource() instanceof ScriptButton)
		{
			selectedButton = (ScriptButton)e.getSource();
			ScriptButton sb = (ScriptButton)e.getSource();
			sbMenu.setEnabled(true);
			sbMenu.show(e.getComponent(), e.getX(), e.getY());
		}
	}
	
	public void addNewButton()
	{
		ScriptButton newButton = addScriptButton("New button", "New action");
	}
	
	/**
	 * Loads the script buttons from the settings file
	 */
	public void loadSettings()
	{
		// Firstly, clear all of the existing buttons
		buttonList.clear();
		removeAll();
		//revalidate();
		
		BufferedReader br;
		String configData = "";
		String []tokens;
		
		try
		{
			br = new BufferedReader(new FileReader(new File(settingsFile)));
		}
		catch(Exception e)
		{
			JOptionPane.showMessageDialog(null, "Failed to open " + settingsFile + "!", "Error", JOptionPane.ERROR_MESSAGE);
			return;
		}
		try
		{
			configData = br.readLine();
		}
		catch(IOException ioe)
		{
			JOptionPane.showMessageDialog(null, "Warning: Exception occurred when reading " + settingsFile + "!", "Error", JOptionPane.ERROR_MESSAGE);
			return;
		}
		while(configData != null)
		{
			try
			{
				if(configData != null && configData.length() > 0 && !configData.trim().startsWith("#"))
				{
					tokens = configData.split("=");
					if(tokens.length > 1)
					{
						// Remove padding spaces
						for(int i=0; i<tokens.length; i++)
							tokens[i] = tokens[i].trim();
						addScriptButton(tokens[0], tokens[1]);	
					}
				}
				
				// Read and parse a line
				configData = br.readLine();
			}
			catch(IOException ioee)
			{
				JOptionPane.showMessageDialog(null, "Warning: Failed to read from " + settingsFile + "!", "Error", JOptionPane.ERROR_MESSAGE);
				return;
			}
		}
		try
		{
			br.close();
		}
		catch(IOException closeEx)
		{
			JOptionPane.showMessageDialog(null, "Warning: Exception occurred when closing " + settingsFile + "!", "Error", JOptionPane.ERROR_MESSAGE);
		}	
	}
	
	/**
	 * Saves the script buttons to the settings file
	 */
	public void saveSettings()
	{
		BufferedWriter bw;
		boolean autoRefresh;
		int refreshDelay;
		
		try
		{
			bw = new BufferedWriter(new FileWriter(new File(settingsFile)));
			bw.write("# SARJ Script button settings\r\n\r\n");
			
			Iterator it = buttonList.iterator();
			ScriptButton sb;
			while(it.hasNext())
			{
				sb = (ScriptButton)it.next();
				bw.write(sb.getName() + "=" + sb.getScriptFile() + "\r\n");
			}
			bw.close();
		}
		catch(IOException ioe)
		{
			JOptionPane.showMessageDialog(null, "Warning: Failed to save configuration settings to " + settingsFile + "!", "Error", JOptionPane.ERROR_MESSAGE);
			return;
		}
	}
}