/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */
 
package sarj;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.border.*;
import java.net.*;
import java.io.*;
import scripting.*;
import FESI.jslib.*;

/**
 * The main SARJ application class
 *
 * @author Richard Davies
 */
public class Sarj extends WindowAdapter implements Runnable, GUIListener, ConnectionListener, ScriptingFacade
{
	////////////////////////////////////////////////////////////////////////////
	// STATICS
	////////////////////////////////////////////////////////////////////////////
	// Config file location
	final static String configFileLocation = "config.txt";
	// Script settings file location
	final static String scriptSettingsFileLocation = "scripts.cfg";
	// Script buttons settings file location
	final static String scriptButtonSettingsFileLocation = "scriptbuttons.cfg";
	// Version string
	final static String VERSION = "1.2.0";
	// Single SARJ instance (Singleton)
	final static Sarj instance = new Sarj();
	// Strings sent by the server that may be of interest (e.g. to cause scripting
	// events to fire)
	final static String SERVMSG_ADMIN_CONNECTED = "Admin connected.";
	final static String SERVMSG_ADMIN_DISCONNECTED = "Admin disconnected.";
	final static String SERVMSG_TIME_LEFT = "Time Left: ";
	final static String SERVMSG_JOIN_REQUEST = " requesting game...";
	final static String SERVMSG_PLAYER_JOINING = " joining game ";
	final static String SERVMSG_PLAYER_LEAVE = " has left the game.";
	final static String SERVMSG_PLAYER_JOINED = " has joined the game.";
	// The interval at which timed scripts are checked
	final static int SCRIPT_TIME_INTERVAL = 1000;
	// A couple of commonly used icons
	public static ImageIcon OK_ICON = new ImageIcon(instance.getClass().getResource("/images/ok.gif"));
	public static ImageIcon CANCEL_ICON = new ImageIcon(instance.getClass().getResource("/images/cancel.gif"));
	////////////////////////////////////////////////////////////////////////////
	
	private String mapName = "?", gameType = "?";
	private int scoreLimit, timeLimit, timeLeft, numPlayers; 
		
	////////////////////////////////////////////////////////////////////////////
	// CLASS VARIABLES
	////////////////////////////////////////////////////////////////////////////
	private int refreshDelay, refreshClock;
	private boolean autoRefresh;
	private GUI myGUI;
	private Connection myConnection;
	private Timer scriptTimer;
	private long timePassed;	// for interval based scripts
	////////////////////////////////////////////////////////////////////////////
	
	/**
	 * The main method
	 */
	public static void main(String []args)
	{
		// Read the UI style from the config file and apply
		try
		{
			UIManager.setLookAndFeel(ConfigDialog.getUIClassNameFromConfigFile(configFileLocation));
		}
		catch(Exception e)
		{
			Sarj.printError("Exception: " + e.getMessage());
		}
		
		//Initialise all scripting interpreters
		try
		{
			Class.forName("scripting.InterpreterDriverManager");
			Class.forName("scripting.JSInterpreterDriver");
		}
		catch(ClassNotFoundException e)
		{
			Sarj.printError("Exception: " + e.getMessage());
			return;
		}
		
		Thread sarjThread = new Thread(instance);
		sarjThread.start();
	}
	
	/**
	 * Default constructor
	 */
	protected void Sarj()
	{
		//
	}
	
	/**
	 * Returns the instance of the SARJ program
	 */
	 public static Sarj instance()
	 {
	 	return instance;
	 }
	 
	/**
	 * Called when the main execution thread is started by a call to Thread.start()
	 */
	public void run()
	{
		JFrame.setDefaultLookAndFeelDecorated(true);
		JDialog.setDefaultLookAndFeelDecorated(true);
		
		myGUI = new GUI(configFileLocation, scriptSettingsFileLocation, scriptButtonSettingsFileLocation, this);
		myConnection = new Connection(this);
		
		refreshClock = 0;
		scriptTimer = new Timer(SCRIPT_TIME_INTERVAL, new ActionListener()
		{
			public void actionPerformed(ActionEvent e)
			{
				checkTimedScripts(SCRIPT_TIME_INTERVAL);
			}
		});	// Check every second for timed scripts
		scriptTimer.start();
		
		timePassed = 0;
		
		myGUI.show();
		
		// Dispatch the onLoad script
		dispatchEventScript("onLoad", null, null);
		
		InputStream is;
		String buffer;
		
		int buff[] = new int[1188];
		int bytesRead=0;
		int b, i;
								
		while(true)
		{
			try { Thread.sleep(50); }
			catch(InterruptedException e) { }
			
			if(!myConnection.isClosed())
			{
				try
				{
					while(true)	// Continually wait for data
					{
						buffer = myConnection.readString();
						if(buffer.startsWith("REFRESH"))
						{
							b = myConnection.read();
							if(b == 0x52)	// 'REFRESH' prepended to actual data
							{
								myConnection.skip(8);
								i=0;
							}
							else
							{
								buff[0] = b;	// Keep the byte
								i=1;
							}		
							for(; i<1188; i++)	// Read rest of refresh packet
							{
								buff[i] = myConnection.read();
							}	
							parseRefresh(buff);
						}
						else
						{	
							// Do some checking here to see if this data
							// should fire off a scripting event
							if(buffer.trim().equals(SERVMSG_ADMIN_CONNECTED))
							{
								dispatchEventScript("onAdminConnect", null, null);
							}
							else if(buffer.trim().equals(SERVMSG_ADMIN_DISCONNECTED))
							{
								// Fire AdminDisconnected() scripting event
								dispatchEventScript("onAdminDisconnect", null, null);
							}
							else if(buffer.startsWith(SERVMSG_TIME_LEFT))
							{
								// Fire TimeLeftNotify() scripting event
								String argValues[] = new String[1];
								argValues[0] = buffer.substring(11, 12);
								argValues[0] = argValues[0].trim();
								
								dispatchEventScript("onTimeLeftNotify", new String[]{"TIME_LEFT"}, argValues);
							}
							else if(buffer.trim().endsWith(SERVMSG_JOIN_REQUEST))
							{
								// Fire JoinRequest() scripting event
								String []argValues = new String[2];
								
								String address = buffer.substring(0, buffer.indexOf(SERVMSG_JOIN_REQUEST));
								address = address.trim();
								String []toks = address.split(":");
								argValues[0] = toks[0];	// IP
								argValues[1] = toks[1]; // Port
								
								dispatchEventScript("onJoinRequest", new String[]{"PLAYER_IP", "PLAYER_PORT"}, argValues);
							}
							else if(buffer.trim().indexOf(SERVMSG_PLAYER_JOINING) != -1)
							{
								// Fire PlayerJoin() scripting event
								String []toks = buffer.trim().split(SERVMSG_PLAYER_JOINING);
								String []argValues = new String[3];
								String []ipAndPort = toks[1].trim().split(":");
								
								argValues[0] = toks[0].trim();	// player name
								argValues[1] = ipAndPort[0].substring(1);	// IP
								argValues[2] = ipAndPort[1].substring(0, ipAndPort[1].length() - 1);
								
								dispatchEventScript("onPlayerJoin", new String[]{"PLAYER_NAME", "PLAYER_IP", "PLAYER_PORT"}, argValues);
							}
							else if(buffer.trim().endsWith(SERVMSG_PLAYER_LEAVE))
							{
								// Fire PlayerLeave() scripting event
								String playerName = buffer.substring(0, buffer.indexOf(SERVMSG_PLAYER_LEAVE));
								String []argValues = new String[1];
								argValues[0] = playerName.trim();
								
								dispatchEventScript("onPlayerLeave", new String[]{"PLAYER_NAME"}, argValues);
							}
							else if(buffer.trim().endsWith(SERVMSG_PLAYER_JOINED))
							{
								// Not implemented
								// Note: This is still here to cope with players with names
								// beginning with '[', otherwise it gets interpreted next
								// as speech
							}
							else if(buffer.trim().startsWith("["))
							{
								// Player has spoken
								// It's actually impossible to correctly extract the
								// player's name, as a name can consist of '] '
								// Therefore, I cycle through all the connected
								// players to see if the string starts with that
								// nickname.
								
								
								int index = buffer.indexOf("] ");
								if(index < 0) return;
								
								String playerData[][] = getPlayerData();
								String playerName;
								String playerText;
								
								if(playerData == null) return;
								
								boolean found=false;
								
								for(int x=0; x<playerData.length; x++)
								{
									if(playerData[x].length < 1) break;
									
									playerName = playerData[x][0];
									
									if(buffer.substring(1).startsWith(playerName))
									{
										found = true;
										
										playerText = buffer.substring(playerName.length() + 3);
										playerText = playerText.trim();
										
										String []argValues = new String[2];
										argValues[0] = playerName;
										argValues[1] = playerText;
								
										dispatchEventScript("onPlayerSpeak", new String[]{"PLAYER_NAME", "SPEECH_TEXT"}, argValues);
									}
								}
								
								if(!found)
								{
									Sarj.printError(buffer);
									Sarj.printError("Player name not found");
								}
							}
			
							myGUI.writeToConsole(buffer);
							
							String []argValues = new String[1];
							argValues[0] = buffer;
							dispatchEventScript("onDataReceived", new String[]{"DATA_STRING"}, argValues);
						}
							
						try { Thread.sleep(50); }
						catch(InterruptedException e) { }
					}
				}
				catch(IOException ioe)
				{
					if(!myConnection.isClosed())
					{
						myConnection.disconnect();
						myGUI.disconnected();
						myGUI.stopRefreshTimer();
					}
				}
			}
		}
	}
	
	/**
	 * Displays error messages
	 *
	 * @param errorString the error message to display
	 */
	 public static void printError(String errorString)
	 {
	 	GUI.printError(errorString);
	 }
	
	/**
	 * Parses a refresh packet sent from the Soldat server containing details of
	 * the current game, and updates the GUI as neccessary. The refresh packet
	 * is sent from the server in response to a REFRESH request made by the
	 * SARJ client, either by clicking the refresh button on the GUI or by the
	 * refresh timer triggering
	 *
	 * @param refreshData the data contained in the refresh packet
	 */
	private synchronized void parseRefresh(int refreshData[])
	{
		int []nameLengthTable = new int[32];	// Holds length of player names
		String []nameList = new String[32];	// Holds player names
		int []playerTeamTable = new int[32];	// Holds player teams
		int []playerScoreTable = new int[32];	// Holds player scores
		int []playerDeathTable = new int[32];	// Holds player deaths
		int []playerIDTable = new int[32];	// Holds player IDs
		int []playerPingTable = new int[32];	// Holds player pings
		String []playerIPList = new String[32];
		int teamScoreTable[] = new int[4];
		int teamTotalDeathsTable[] = new int[4];	// Total deaths for each team
		int teamTotalScoreTable[] = new int[4]; // Total score of each team
		int teamPingTable[] = new int[4];	// Average ping for each team
		int offset;
		int i, j;
		char buff[];
		
		if(refreshData.length < 1188) return;	// Incomplete packet
		
		for(i=0; i<32; i++)
		{
			nameList[i] = new String();
		}
		
		for(i=0; i<4; i++)
		{
			teamTotalDeathsTable[i] = 0;
			teamTotalScoreTable[i] = 0;
			teamPingTable[i] = 0;
		}
		
		numPlayers = 0;
		offset = 0x320;	// Player teams
		while(refreshData[offset] != 0xFF && numPlayers < 32)
		{
			playerTeamTable[numPlayers] = refreshData[offset];
			numPlayers++;
			offset++;
		}
		
		offset=0x00;
		for(i=0; i<numPlayers; i++)
		{
			// Extract length of name
			nameLengthTable[i] = refreshData[offset];
			offset++;
			if(nameLengthTable[i] < 1) break;	// Reached end of names
			// Extract name
			buff = new char[nameLengthTable[i]];
			for(j=0; j<nameLengthTable[i]; j++)
			{
				buff[j] = (char)refreshData[offset];
				offset++;
			}
			for(;j<24; j++)
				offset++;
			
			nameList[i] = new String(buff);
		}
		
		// Time left and Time limit
		
		// The time limit is stored in 4 bytes in backwards order
		// This value must be divided by 0x3c to get the time limit in seconds 
		offset = 0x499;
		timeLimit = (refreshData[offset+3] << 24 |
						refreshData[offset+2] << 16 |
						refreshData[offset+1] << 8 |
						refreshData[offset]) / 0x3c;
		
		// The time left is also stored in 4 bytes, in backwards order
		offset = 0x49d;
		timeLeft = (refreshData[offset+3] << 24 |
						refreshData[offset+2] << 16 |
						refreshData[offset+1] << 8 |
						refreshData[offset]) / 0x3c;	// to get in seconds
								
		offset = 0x340;	// Player scores
		for(i=0; i<numPlayers; i++)
		{
			playerScoreTable[i] = refreshData[offset];
			playerScoreTable[i] |= (refreshData[offset+1] << 8);
			offset+=2;
		}
			
		offset = 0x380;	// Player deaths
		for(i=0; i<numPlayers; i++)
		{
			playerDeathTable[i] = refreshData[offset];
			offset+=2;
		}
		
		offset = 0x3C0;	// Player pings
		for(i=0; i<numPlayers; i++)
		{
			playerPingTable[i] = refreshData[offset];
			offset++;
		}
			
		offset = 0x3E0;	// Player IDs
		for(i=0; i<numPlayers; i++)
		{
			playerIDTable[i] = refreshData[offset];
			offset++;
		}
			
		offset = 0x400;	// Player IPs
		for(i=0; i<numPlayers; i++)
		{
			int ip[] = new int[4];
			for(j=0; j<4; j++)
			{
				ip[j] = refreshData[offset];
				offset++;
			}
			playerIPList[i] = ip[0] + "." + ip[1] + "." + ip[2] + "." + ip[3];
			
		}
		
		offset = 0x4A1;	// Game score limit
		scoreLimit = refreshData[offset];
		
		// Game type
		offset = 0x4A3;	// Game mode/type
		int type = refreshData[offset];
		switch(type)
		{
			case 0:
				gameType = "Deathmatch";
				break;
			case 1:
				gameType = "Pointmatch";
				break;
			case 2:
				gameType = "Teammatch";
				break;
			case 3:
				gameType = "CTF";
				break;
			case 4:
				gameType = "Rambomatch";
				break;
			case 5:
				gameType = "Infiltration";
				break;
			default:
				gameType = "Unknown";
				break;
		}
		
		offset = 0x480;	// Team scores
		for(i=0; i<4; i++)
		{
			// Score is two bytes in network order
			teamScoreTable[i] = (refreshData[offset] | (refreshData[offset+1] << 8));
			offset+=2;
		}
		
		offset = 0x488;	// Map name
		int temp = refreshData[offset];
		int mapNameLength = Integer.parseInt(new Integer(temp).toString());
		
		buff = new char[mapNameLength];
		for(i=0; i<mapNameLength; i++)
		{
			offset++;
			buff[i] = (char)refreshData[offset];
		}
		mapName = new String(buff);
		
		// Refresh the table with the player details
		myGUI.setTitle("SARJ " + VERSION + " :: Connected [" + numPlayers + " players]");
		myGUI.clearPlayerTable();
		for(int x=0; x<numPlayers; x++)
		{
			float ratio;
			String ratioString;
			if(playerScoreTable[x] == 0)
				ratioString = "0";
			else if(playerDeathTable[x] == 0)
				ratioString = "*";
			else
			{
				ratio = (float)((float)playerScoreTable[x] / (float)playerDeathTable[x]);
				ratio = Math.round(ratio*100)/100f;	// Get float to 2 d.p.
				ratioString = (new Float(ratio)).toString();
			}
			myGUI.addPlayer(nameList[x], playerScoreTable[x], playerDeathTable[x], ratioString, playerPingTable[x], playerTeamTable[x], playerIPList[x], playerIDTable[x]);
		}
		
		// Calculate team totals
		int teamPlayerCount[] = new int[4];	// Keeps track of how many players on each team
		for(i=0; i<4; i++)
			teamPlayerCount[i] = 0;
			
		for(i=0; i<numPlayers; i++)
		{
			if(playerTeamTable[i] < 5 && playerTeamTable[i] > 0)	// Player's team is alpha...delta
			{
				teamPlayerCount[playerTeamTable[i]-1]++;	// Increase player count for team
				teamTotalScoreTable[playerTeamTable[i]-1]+=playerScoreTable[i];
				teamTotalDeathsTable[playerTeamTable[i]-1]+=playerDeathTable[i];
				teamPingTable[playerTeamTable[i]-1]+=playerPingTable[i];
			}
		}
		
		for(i=0; i<4; i++)
			if(teamPlayerCount[i] > 0)
				teamPingTable[i] /= teamPlayerCount[i];	// Calculate average
		
		
		// Refresh the table with the team details
		myGUI.clearTeamTable();
		myGUI.addTeam("<html><font color='#CC0000'>Alpha</font></html>", teamScoreTable[0], teamTotalScoreTable[0], teamTotalDeathsTable[0], teamPingTable[0]);
		myGUI.addTeam("<html><font color='#0000CC'>Bravo</font></html>", teamScoreTable[1], teamTotalScoreTable[1], teamTotalDeathsTable[1], teamPingTable[1]);
		myGUI.addTeam("<html><font color='#AAAA00'>Charlie</font></html>", teamScoreTable[2], teamTotalScoreTable[2], teamTotalDeathsTable[2], teamPingTable[2]);
		myGUI.addTeam("<html><font color='#00BB00'>Delta</font></html>", teamScoreTable[3], teamTotalScoreTable[3], teamTotalDeathsTable[3], teamPingTable[3]);
		
		myGUI.setMapName("<html>Map: <font color = '#0000FF'><b>" + mapName + "</b></font></html>");
		myGUI.setTimeLimit("<html>Time limit: <font color = '#0000FF'><b>" + formatTime(timeLimit) + "</b></font></html>");
		myGUI.setTimeLeft("<html>Time left: <font color = '#0000FF'><b>" + formatTime(timeLeft) + "</b></font></html>");
		myGUI.setScoreLimit("<html>Score limit: <font color = '#0000FF'><b>" + scoreLimit + "</b></font></html>");
		myGUI.setGameType("<html>Game: <font color = '#0000FF'><b>" + gameType + "</b></font></html>");
	}

	/**
	 * Called when the Connect button is clicked on the GUI
	 *
	 * @param address	the address of the Soldat server, in dotted IP format
	 *					(X.X.X.X)
	 * @param port		the port number of the Soldat server
	 * @param password	the remote administration password for the server
	 */ 
	public void onConnect(String address, int port, String password)	// From GUI
	{
		try
		{
			myGUI.writeToConsole("Connecting...\n");
			myConnection.connect(address, port);
		}
		catch(UnknownHostException uhe)
		{
			myGUI.writeToConsole("Unknown host: '" + address + "'\n");
		}
		catch(IOException ioe)
		{
			myGUI.writeToConsole("Failed to connect to '" + address + "'\n");
		}
	}
	
	/**
	 * Called when the Disconnect button is clicked on the GUI
	 */
	public void onDisconnect()	// from GUI
	{
		myConnection.disconnect();
		myGUI.disconnected();
	}
	
	/**
	 * Called when the Exit button is clicked on the GUI, or the main GUI window
	 * is closed
	 */
	public void onExit()	// from GUI
	{
		if(myConnection.isClosed() || JOptionPane.showConfirmDialog(myGUI.getMainFrame(), "You are still connected. Exit anyway?", "Exit SARJ?", JOptionPane.YES_NO_OPTION, JOptionPane.QUESTION_MESSAGE) == JOptionPane.YES_OPTION)
		{
			try { myConnection.disconnect(); }
			catch(Exception e) { }
			dispatchEventScript("onExit", null, null);
			System.exit(0);
		}
	}
	
	/**
	 * Called when the Perform button is clicked on the GUI
	 *
	 * @param actionID the ID of the action to be performed
	 */
	public void onAction(int actionID)
	{
		String data;
		
		switch(actionID)
		{
			case 0:	// Shutdown server
				if(JOptionPane.showConfirmDialog(myGUI.getMainFrame(), "Really shutdown the server?", "Shutdown server", JOptionPane.OK_CANCEL_OPTION, JOptionPane.QUESTION_MESSAGE) == JOptionPane.OK_OPTION)
				{
					myConnection.sendData("SHUTDOWN\n");
					myConnection.disconnect();
				}
				break;
			case 1:	// Send server message
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Server message:", "Server message", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					myConnection.sendData("/say " + data + "\n");
				}	
				break;
			case 2:	// Add bot
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Name of bot to add:", "Add bot", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					String []addOptions = {"Alpha", "Bravo", "Charlie", "Delta"};
				
					int addResult = JOptionPane.showOptionDialog(myGUI.getMainFrame(), "Please select a team:", "Add bot", JOptionPane.YES_NO_OPTION, JOptionPane.PLAIN_MESSAGE, null, addOptions, null);
					if(addResult != JOptionPane.CLOSED_OPTION)
					{
					
						myConnection.sendData("/addbot" + (addResult+1) + " " + data + "\n");
					}
				}					
				break;
			case 3:	// Kick player
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Player name/number:", "Kick player", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					myConnection.sendData("/kick " + data + "\n");
				}
				break;
			case 4:	// Ban player
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Player name/number:", "Ban player", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					myConnection.sendData("/ban " + data + "\n");
				}				
				break;
			case 5:	// Ban IP
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "IP:", "Ban IP", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					// Check for valid IP
					int numDots=0;
					for(int i=0; i<data.length(); i++)
						if(data.charAt(i) == '.') numDots++;
					if(numDots != 3)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, that isn't a valid IP address", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}
					myConnection.sendData("/banip " + data + "\n");
				}				
				break;
			case 6:	// Unban IP
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "IP:", "Unban IP", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					// Check for valid IP
					int numDots=0;
					for(int i=0; i<data.length(); i++)
						if(data.charAt(i) == '.') numDots++;
					if(numDots != 3)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, that isn't a valid IP address", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}	
					myConnection.sendData("/unbanip " + data + "\n");
				}		
				break;
			case 7:	// Change map
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "New map:", "Change map", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					myConnection.sendData("/map " + data + "\n");
				}		
				break;
			case 8:	// Restart match	
				myConnection.sendData("/restart\n");	
				break;
			case 9:	// Load next map	
				myConnection.sendData("/nextmap\n");
				break;
			case 10:	// Add remote admin
				String []addOptions = {"Add by name", "Add by IP"};
					
				int addResult = JOptionPane.showOptionDialog(myGUI.getMainFrame(), "Please select an option:", "Add remote admin", JOptionPane.YES_NO_OPTION, JOptionPane.PLAIN_MESSAGE, null, addOptions, null);
				if(addResult != JOptionPane.CLOSED_OPTION)
				{
					String input, dataString = null;
						
					if(addResult == 0)	// Add by name
					{
						input = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Player name:", "Add remote admin", JOptionPane.PLAIN_MESSAGE);
						if(input != null && input.length() > 0)
						{
							dataString = "/adm " + input + "\n";
						}
					}
					else
					{
						input = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Player IP:", "Add remote admin", JOptionPane.PLAIN_MESSAGE);
						if(input != null && input.length() > 0)
						{
							// Check for valid IP
							int numDots=0;
							for(int i=0; i<input.length(); i++)
								if(input.charAt(i) == '.') numDots++;
							if(numDots != 3)
							{
								JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, that isn't a valid IP address", "Error", JOptionPane.ERROR_MESSAGE);
								return;
							}
							dataString = "/admip " + input + "\n";
						}
					}
					myConnection.sendData(dataString);	
				}
				break;
			case 11:	// Remove remote admin
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Player IP:", "Remove remote admin", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					// Check for valid IP
					int numDots=0;
					for(int i=0; i<data.length(); i++)
					if(data.charAt(i) == '.') numDots++;
					if(numDots != 3)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, that isn't a valid IP address", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}
					myConnection.sendData("/unadm " + data + "\n");
				}	
				break;
			case 12:	// Kick last player
				myConnection.sendData("/kicklast\n");
				break;
			case 13:	// Set respawn time
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Respawn time (seconds):", "Set respawn time", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					int respawnTime;
					try
					{
						respawnTime = Integer.parseInt(data);
					}
					catch(NumberFormatException nfe)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, that isn't a valid respawn time", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}
					if(respawnTime < 1 || respawnTime > 30)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, respawn time must be between 1 and 30 seconds", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}	
					myConnection.sendData("/respawntime " + data + "\n");	
				}
				break;
			case 14:	// Set max respawn time
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Maximum respawn time (seconds):", "Set maximum respawn time", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					int respawnTime;
					try
					{
						respawnTime = Integer.parseInt(data);
					}
					catch(NumberFormatException nfe)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, that isn't a valid respawn time", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}
					if(respawnTime < 1 || respawnTime > 30)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, respawn time must be between 1 and 30 seconds", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}	
					myConnection.sendData("/maxrespawntime " + data + "\n");
				}
				break;
			case 15:	// Set kill/point/cap limit
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Kill/point/capture limit:", "Set kill/point/capture limit", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					int limit;
					try
					{
						limit = Integer.parseInt(data);
					}
					catch(NumberFormatException nfe)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, that isn't a valid limit", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}
					if(limit < 1 || limit > 30)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, limit must be between 1 and 30", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}	
					myConnection.sendData("/limit " + data + "\n");
				}
				break;
			case 16:	// Set time limit
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Time limit (minutes):", "Set time limit", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					int limit;
					try
					{
						limit = Integer.parseInt(data);
					}
					catch(NumberFormatException nfe)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, that isn't a valid time limit", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}
					if(limit < 1 || limit > 30)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, time limit must be between 1 and 30 minutes", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}	
					myConnection.sendData("/timelimit " + data + "\n");
				}
				break;
			case 17:	// Set game password
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Player limit:", "Set player limit", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					int limit;
					try
					{
						limit = Integer.parseInt(data);
					}
					catch(NumberFormatException nfe)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, that isn't a valid limit", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}
					if(limit < 1 || limit > 32)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, limit must be between 1 and 32", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}	
					myConnection.sendData("/maxplayers " + data + "\n");
				}
				break;
			case 18:	// Set game password
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "New game password:", "Set game password", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					myConnection.sendData("/password " + data + "\n");
				}
				break;
			case 19:	// Set admin password
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "New admin password:", "Set admin password", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					myConnection.sendData("/adminpass " + data + "\n");
				}
				break;
			case 20:	// Set friendly fire
				String []options = {"On", "Off"};
				int result = JOptionPane.showOptionDialog(myGUI.getMainFrame(), "Select friendly fire state", "Friendly fire", JOptionPane.YES_NO_OPTION, JOptionPane.PLAIN_MESSAGE, null, options, "Off");
				if(result != JOptionPane.CLOSED_OPTION)
					{
					String dataString;
					
					if(result == 0)	// On
					{
						dataString = "/friendlyfire 1\n";
					}
					else
						dataString = "/friendlyfire 0\n";
					
					myConnection.sendData(dataString);	
				}
				break;
			case 21:	// Set vote percentage
				data = JOptionPane.showInputDialog(myGUI.getMainFrame(), "Vote percentage:", "Set vote percentage", JOptionPane.PLAIN_MESSAGE);
				if(data != null && data.length() > 0)
				{
					int percentage;
					try
					{
						percentage = Integer.parseInt(data);
						}
					catch(NumberFormatException nfe)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, that isn't a valid percentage", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}
					if(percentage < 1 || percentage > 100)
					{
						JOptionPane.showMessageDialog(myGUI.getMainFrame(), "Sorry, percentage must be between 1 and 100", "Error", JOptionPane.ERROR_MESSAGE);
						return;
					}	
					myConnection.sendData("/vote% " + data + "\n");
				}
				break;
			default:
				break;
		}
	}
	
	/**
	 * Called when the Refresh button is clicked on the GUI, or the refresh
	 * timer triggers
	 */
	public void onRefresh()	// from GUI button or timer
	{
		myConnection.sendData("REFRESH\n");
	}
	
	/**
	 * Called when a command is sent from the user
	 *
	 * @param command the command string issued by the user
	 */
	public void onCommand(String command)
	{
		myConnection.sendData(command);
	}
	
	/**
	 * Called when the user chooses to kick a player from the list of currently
	 * connected players
	 *
	 * @param playerNumber the unique in-game ID number of the player
	 */
	public void onKickPlayer(int playerNumber)
	{
		myConnection.sendData("/kick " + playerNumber + "\n");
	}
	
	/**
	 * Called when the user chooses to ban a player from the list of currently
	 * connected players
	 *
	 * @param playerNumber the unique in-game ID number of the player
	 */
	public void onBanPlayer(int playerNumber)
	{
		myConnection.sendData("/ban " + playerNumber + "\n");
	}
	
	/**
	 * Called when the user chooses to add a player to the remote administrators
	 * list
	 *
	 * @param playerIP the IP address of the player
	 */
	public void onAddAdmin(String playerIP)
	{
		myConnection.sendData("/admip " + playerIP + "\n");
	}
	
	/**
	 * Called when the user chooses to remove a player from the remote
	 * administrators list
	 *
	 * @param playerIP the IP address of the player
	 */
	public void onRemoveAdmin(String playerIP)
	{
		myConnection.sendData("/unadm " + playerIP + "\n");
	}
	
	/**
	 * Called when the user chooses to assign a connected player to a particular
	 * team
	 *
	 * @param playerNumber the unique in-game ID number of the player
	 * @param teamNumber the ID of the team to assign the player to, where
	 *					 1 = alpha, 2 = beta, 3 = charlie and 4 = delta
	 */
	public void onSetTeam(int playerNumber, int teamNumber)
	{
		myConnection.sendData("/setteam" + teamNumber + " " + playerNumber + "\n");
	}
	
	/**
	 * Called when the user clicks on a ScriptButton
	 *
	 * @param name the name/caption of the button
	 * @param scriptFile the script file associated with the button
	 */
	 public void onScriptButtonClick(String name, String scriptFile)
	 {
	 	try
	 	{
	 		String []argNames = new String[]{"Richard"};
	 		String []argValues = new String[]{"Davies"};
	 		InterpreterDriverManager.executeScriptFile(scriptFile, argNames, argValues);
	 	}
	 	catch(InterpreterDriver.InterpreterException ex)
	 	{
	 		Sarj.printError("Exception while executing script '" + scriptFile + "' : " + ex);
	 	}
	 }
	
	/**
	 * Called when a connection is successfully established with a server, as a
	 * result of calling the connect method of a {@link Connection} object.
	 */
	public void onConnected()	// Called by the Connection object
	{
		myConnection.sendData(myGUI.getPasswordField() + "\n");
		myConnection.sendData("REFRESH\n");
		myGUI.connected();
		dispatchEventScript("onConnect", null, null);
	}
	
	/**
	 * Called when a connection to a server is lost, either gracefully or due to
	 * error
	 */
	public void onDisconnected()
	{
		dispatchEventScript("onDisconnect", null, null);	
	}
	
	/**
	 * Called when a connection error occurrs
	 *
	 * @param message the error message
	 */
	public void onError(String message)
	{
		myGUI.writeToConsole("Connection error: " + message + "\n");
		//myGUI.disconnected();
	}
	
	/**
	 * Formats seconds into mm:ss format
	 *
	 * @param seconds the time to translate, in second
	 */
	public static String formatTime(int seconds)
	{
		if(seconds < 0)
			return("00:00");
			
		int mins = seconds / 60;
		int secs = seconds % 60;
		
		String result = (mins < 10? ("0" + mins + ":") : (mins + ":"));
		
		if(secs < 10)
			result += "0" + secs;
		else
			result += secs;
			
		return result; 
	}
	
	/**
	 * Checks all scripts that run at time intervals or at specific times to see
	 * if they should now execute
	 *
	 * @param interval the amount of time passed, in milliseconds, since the last
	 *				   check.
	 */
	private void checkTimedScripts(int interval)
	{
		timePassed += interval;
		
		String []scripts = myGUI.checkTimedScripts(timePassed);
		
		if(scripts == null) return;
		
		for(int i=0; i<scripts.length; i++)
		{
			try
			{
				InterpreterDriverManager.executeScriptFile(scripts[i], null, null);
			}
			catch(InterpreterDriver.InterpreterException e)
			{
				Sarj.printError("Interpreter exception: " + e);
			}
		}
	}
	
	/**
	 * Handles the executing of the script from the event name
	 *
	 * @param eventName the name of the event
	 * @param argNames an array of argument names to send to the script
	 * @param argValues an array of argument values to send to the script. This
	 *					must be the same size as the argNames array.
	 */
	private void dispatchEventScript(String eventName, String []argNames, String []argValues)
	{
		try
		{
			String scriptFile = myGUI.getFilenameForEvent(eventName);
			
			if(myGUI.isEventEnabled(eventName))
			{
				// Execute the script in filename
				InterpreterDriverManager.executeScriptFile(scriptFile, argNames, argValues);
			}
		}
		catch(EventNotFoundException ex1)
		{
			Sarj.printError("Event not found exception: " + ex1);
		}
		catch(InterpreterDriver.InterpreterException ex2)
		{
			Sarj.printError("Interpreter exception: " + ex2);
		}
	}
	
	/**
	 * Runs the specified script file
	 *
	 * @param scriptFile the path of the script file to execute
	 */
	public void dispatchScript(String scriptFile)
	{
		try
		{
			InterpreterDriverManager.executeScriptFile(scriptFile, null, null);
		}
		catch(InterpreterDriver.InterpreterException ex)
		{
			Sarj.printError("Interpreter exception: " + ex);
		}
	}
	
	////////////////////////////////////////////////////////////////////////////
	// SCRIPTING FACADE INTERFACE METHODS
	////////////////////////////////////////////////////////////////////////////
	// The following methods are simply to provide an API for scripting support
	
	/**
	 * Returns the name of the map currently being played on the server
	 *
	 * @return a {@link String} containing the name of the current map
	 */
	public String getCurrentMap()
	{
		return mapName;
	}
	
	/**
	 * Returns the type of game running on the server
	 *
	 * @return a {@link String} containing the name of the current game type
	 */
	public String getGameType()
	{
		return gameType;
	}
	
	/**
	 * Returns the time limit of the current game
	 *
	 * @return the current time limit, in seconds
	 */
	public int getTimeLimit()
	{
		return timeLimit;
	}
	
	/**
	 * Returns the time left within the current game
	 *
	 * @return the time remaining, in seconds
	 */
	public int getTimeLeft()
	{
		return timeLeft;
	}
	
	/**
	 * Returns the score limit of the game
	 *
	 * @return the score limit
	 */
	public int getScoreLimit()
	{
		return scoreLimit;
	}
	
	/**
	 * Returns the number of players connected to the server
	 *
	 * @return the number of players currently connected
	 */
	public int getNumPlayers()
	{
		return numPlayers;
	}
	
	/**
	 * Sets the main SARJ window title to that specified
	 *
	 * @param title the new title for the window
	 */
	public void setWindowTitle(String title)
	{
		myGUI.getMainFrame().setTitle(title);
	}
	
	/**
	 * Returns the main SARJ window title
	 *
	 * @return a {@link String} containing the title of the main window
	 */
	 public String getWindowTitle()
	 {
	 	return myGUI.getMainFrame().getTitle();
	 }
	
	/**
	 * Returns the value of the IP field on the main window
	 *
	 * @return a {@link String} containing the value of the IP field
	 */
	public String getIPField()
	{
		return myGUI.getIPField();
	}
	
	/**
	 * Returns the value of the port field on the main window
	 *
	 * @return a {@link String} containing the value of the port field
	 */
	public String getPortField()
	{
		return myGUI.getPortField();
	}
	
	/**
	 * Returns the current value stored in the password field on the main window
	 *
	 * @return a String object containing the unmasked value of the password field
	 */
	public String getPasswordField()
	{
		return myGUI.getPasswordField();
	}
	
	/**
	 * Returns the value of the command combo box on the main window
	 *
	 * @return a {@link String} containing the value of the command combo box
	 */
	public String getCommandBox()
	{
		return myGUI.getCommandBox();
	}
	
	/**
	 * Sets the IP field on the main window
	 *
	 * @param ip the new IP address value
	 */
	public void setIPField(String ip)
	{
		myGUI.setIPField(ip);
	}
	
	/**
	 * Sets the port field on the main window
	 *
	 * @param port the new port value
	 */
	public void setPortField(String port)
	{
		myGUI.setPortField(port);
	}
	
	/**
	 * Sets the password field on the main window
	 *
	 * @param password the new password value
	 */
	public void setPasswordField(String password)
	{
		myGUI.setPasswordField(password);
	}
	
	/**
	 * Sets the command combo box field on the main window
	 *
	 * @param command the new value to set the command combo box to
	 */
	public void setCommandBox(String command)
	{
		myGUI.setCommandBox(command);
	}
	
	/**
	 * Clears the console on the main window
	 */
	public void clearConsole()
	{
		myGUI.clearConsole();
	}
	
	/**
	 * Disconnects the client from any connected server
	 */
	public void disconnectClient()
	{
		myConnection.disconnect();
		myGUI.disconnected();
	}
	
	/**
	 * Connects the client to the specified IP and port
	 *
	 * @param ip the IP address of the server
	 * @param port the port number of the server
	 */
	public void connectClient(String ip, int port)
	{
		try { myConnection.connect(ip, port); }
		catch(Exception ex)
		{
			Sarj.printError("Exception: " + ex);
		}
	}
	
	/**
	 * Connects the client to the IP and port displayed on the main window
	 */
	public void connectClient()
	{
		try { myConnection.connect(getIPField(), new Integer(getPortField()).intValue()); }
		catch(Exception ex)
		{
			Sarj.printError("Exception: " + ex);
		}
	}
	
	/**
	 * Writes text to the main command console
	 *
	 * @param text the text to print on the console
	 */
	public void writeToConsole(String text)
	{
		myGUI.writeToConsole(text);
	}
	
	/**
	 * Sends data to a connected server
	 *
	 * @param data the data to send
	 */
	public void sendData(String data)
	{
		myConnection.sendData(data);
	}
	
	/**
	 * Returns the row of the player table with the player name matching that
	 * provided
	 *
	 * @param playerName the name of the player to match
	 * @return the data contained in the table relating to the player
	 */
	public String[] getPlayerData(String playerName)
	{
		Object []tableData = myGUI.getPlayerData(playerName);
		
		String []result = new String[tableData.length];
		for(int i=0; i<result.length; i++)
		{
			result[i] = tableData[i].toString();
		}
		
		return result;
	}
	
	/**
	 * Returns all data within the player table on the main window
	 *
	 * @return a 2D array of {@link String} objects containing the data in the
	 *		   table
	 */
	public String[][] getPlayerData()
	{
		Object [][]tableData = myGUI.getPlayerData();
		
		if(tableData.length > 0)
		{
			String [][]result = new String[tableData.length][tableData[0].length];
			
			for(int j=0; j<result.length; j++)
				for(int i=0; i<result[0].length; i++)
				{
					result[j][i] = tableData[j][i].toString();
				}
				
			return result;
		}
		
		return new String[][]{{}};
	}
	
	/**
	 * Returns the row of the team table with the team name matching that
	 * provided
	 *
	 * @param teamName the name of the team to match
	 * @return the data contained in the table relating to the team
	 */
	public String[] getTeamData(String teamName)
	{
		Object []tableData = myGUI.getTeamData(teamName);
		
		String []result = new String[tableData.length];
		for(int i=0; i<result.length; i++)
		{
			result[i] = tableData[i].toString();
		}
		
		return result;
	}
	
	/**
	 * Returns all data within the team table on the main window
	 *
	 * @return a 2D array of {@link String} objects containing the data in the
	 *		   table
	 */
	public String[][] getTeamData()
	{
		Object [][]tableData = myGUI.getTeamData();
		
		if(tableData.length > 0)
		{
			String [][]result = new String[tableData.length][tableData[0].length];
			
			for(int j=0; j<result.length; j++)
				for(int i=0; i<result[0].length; i++)
				{
					result[j][i] = tableData[j][i].toString();
				}
				
			return result;
		}
		
		return new String[][]{{}};
	}
}