/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */
 
package sarj;

import javax.swing.*;
import javax.swing.table.*;
import java.util.*;

/**
 * The PlayerTableModel class defines a model to use with the player table on the
 * main GUI window
 */
class PlayerTableModel extends AbstractTableModel
{
	private String[] columns = {"Name", "Score", "Deaths", "Ratio", "Ping", "Team", "IP", "#"};
	private Vector<Object[]> playerData;
	
	private static final long serialVersionUID = 5947847346149275958L;
	
	/**
	 * Creates a new PlayerTableModel object
	 */
	public PlayerTableModel()
	{
		playerData = new Vector<Object[]>();
	}
	
	/**
	 * Returns the number of rows currently in the table
	 *
	 * @return the number of rows currently in the table
	 */
	public int getRowCount()
	{
		return playerData.size();
	}
	
	/**
	 * Returns the number of columns currently in the table
	 *
	 * @return the number of columns currently in the table
	 */
	public int getColumnCount()
	{
		return columns.length;
	}
	
	/**
	 * Returns all the data within the table
	 *
	 * @return a 2D array of objects containing the data within the table fields
	 */
	 public Object[][] getData()
	 {
	 	int rowCount = getRowCount();
	 	int colCount = getColumnCount();
	 	Object [][]data = new Object[rowCount][colCount];
	 	
	 	if(rowCount == 0)
	 		return new Object[][]{{}};
	 	
	 	for(int i=0; i<rowCount; i++)
	 	{
	 		data[i] = getRow(i);
	 	}
	 	
	 	return data;
	 }
	
	/**
	 * Returns the items in a row of the table
	 *
	 * @param row the 0-based index of the row to retrieve
	 *
	 * @return an array of objects containing the elements in each cell in the
	 *			specified row
	 */
	public synchronized Object []getRow(int row)
	{
		if(row < playerData.size())
		{
			Object tableData[] = (Object[])playerData.get(row);
			
			// copy data to a new object
			Object []result = tableData.clone();
			
			// Filter out the HTML tags
			
			// Filter out HTML from ping field
			String field = (String)result[4];
			String []toks = field.split("[<>]");	// Separate by HTML tags
			
			result[4] = toks[4];
			
			// Filter out HTML from team field
			field = (String)result[5];
			toks = field.split("[<>]");	// Separate by HTML tags
			result[5] = toks[4];
			
			return result;
		}
		else return (Object[])null;
	}
	
	/**
	 * Returns an element in a cell of the table
	 *
	 * @param row the 0-based index of the row of the element to retrieve
	 * @param column the 0-based index of the column of the element to retrieve
	 *
	 * @return an object containing the element in the specified row and column
	 */
	public Object getValueAt(int row, int column)
	{
		if(row < playerData.size())
		{
			Object []pe = (Object [])playerData.get(row);
			return pe[column];
		}
		else return (Object[])null;
	}
	
	/**
	 * Returns the class of the object type of a column
	 *
	 * @param column the 0-based index of the column
	 *
	 * @return the class of objects in the specified column
	 */
	public Class getColumnClass(int column)
	{
		Object obj = getValueAt(0, column);
		if(obj != null)
        	return obj.getClass();
        else return (Class)null;
    }
    
    /**
     * Returns the name of a column
     *
     * @param column the 0-based index of the column
     *
     * @return a String containing the name of the column
     */
    public String getColumnName(int column)
    {
    	if(column < 8)
        	return columns[column];
        else return "Unknown";
    }
    
    /**
     * Clears all data held within the table
     */
    public synchronized void clear()
	{
		playerData.clear();
		fireTableDataChanged();
	}
	
	/**
	 * Adds details of a player to the table
	 *
	 * @param name the name of the player
	 * @param score the player's score
	 * @param deaths the player's number of deaths
	 * @param ratio the player's kill-to-death ratio
	 * @param ping the player's ping
	 * @param team the player's team number (1 = Alpha, 2 = Beta, 3 = Charlie,
	 *			   4 = Delta)
	 * @param ip the player's IP address
	 * @param num the player's unique in-game ID number
	 */
	public synchronized void addPlayer(String name, int score, int deaths, String ratio, int ping, int team, String ip, int num)
	{
		Object newEntry[] = new Object[8];
		newEntry[0] = name;
		newEntry[1] = new Integer(score);
		newEntry[2] = new Integer(deaths);
		newEntry[3] = ratio;
		String pingColorString;
		
		if(ping < 101)
			pingColorString = "#339900";	// Low ping
		else if(ping < 200)
			pingColorString = "#CC9900";	// Medium ping
		else
			pingColorString = "#CC0000";	// High ping
			
		newEntry[4] = "<html><font color=" + pingColorString + ">" + ping + "</font></html>";
		
		switch(team)
		{
			case 1:
				newEntry[5] = "<html><font color='#CC0000'>Alpha</font></html>";
				break;
			case 2:
				newEntry[5] = "<html><font color='#0000CC'>Bravo</font></html>";
				break;
			case 3:
				newEntry[5] = "<html><font color='#AAAA00'>Charlie</font></html>";
				break;
			case 4:
				newEntry[5] = "<html><font color='#00BB00'>Delta</font></html>";
				break;
			case 5:
				newEntry[5] = "<html><font color='#000000'>Spectator</font></html>";
				break;
			default:
				newEntry[5] = "<html><font color='#000000'>None</font></html>";
				break;
		}
		
		newEntry[6] = ip;
		newEntry[7] = new Integer(num);
		
		playerData.add(newEntry);
		fireTableDataChanged();
	}
}