/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */

package sarj;

import javax.swing.*;
import java.awt.event.*;
import java.awt.*;
import java.io.*;
import java.util.Calendar;

/**
 * The log console dialog for displaying error messages
 */
public class LogDialog extends JDialog implements ActionListener, KeyListener
{
	private static final long serialVersionUID = 5947847346149275958L;
	
	private JTextArea logArea;
	private JButton closeButton;
	
	/**
	 * Default constructor
	 */
	public LogDialog()
	{
		super();
		
		setTitle("SARJ log console");
		setFocusable(true);
		addKeyListener(this);
		
		logArea = new JTextArea(15, 80);
		logArea.setEditable(false);
		logArea.setFont(new Font("Lucida console", Font.PLAIN, 12));
		
		closeButton = new JButton("Close");
		closeButton.setToolTipText("Close the log console");
		closeButton.addActionListener(this);
		closeButton.setAlignmentX(Component.CENTER_ALIGNMENT);
		
		setLayout(new BoxLayout(getContentPane(), BoxLayout.PAGE_AXIS));
		add(new JScrollPane(logArea));
		add(Box.createRigidArea(new Dimension(0, 5)));
		add(closeButton);
		add(Box.createRigidArea(new Dimension(0, 5)));
		
		pack();
		
		setLocationRelativeTo(null);
	}
	
	/**
	 * Prints an error to the console area and to standard err
	 *
	 * @param errorString the error message to print
	 */
	public void printError(String errorString)
	{
		Calendar cal = Calendar.getInstance();
		
		int hour, minute;
		String timeStamp = "[";
		hour = cal.get(Calendar.HOUR_OF_DAY);
		minute = cal.get(Calendar.MINUTE);
		timeStamp += (hour < 10 ? "0" + new Integer(hour).toString() : new Integer(hour).toString());
		timeStamp += ":";
		timeStamp += (minute < 10 ? "0" + new Integer(minute).toString() : new Integer(minute).toString());
		timeStamp += "] ";
		
		logArea.append(timeStamp + errorString + "\n");
		// Set scrollbar position to end of window
		logArea.setCaretPosition(logArea.getDocument().getLength());
		System.err.println(timeStamp + errorString);
	}
	
	/**
	 * Called as a result of implementing ActionListener
	 *
	 * @param e an object containing details of the event
	 */
	public void actionPerformed(ActionEvent e)
	{
		Object source = e.getSource();
		
		if(source == closeButton)
		{
			setVisible(false);
		}
	}
	
	/**
	 * Called as a result of implementing KeyListener
	 *
	 * @param e object containing details of the key event
	 */
	public void keyPressed(KeyEvent e)
	{
		if((e.getKeyCode() == KeyEvent.VK_S) && (e.isAltDown()))
		{
			setVisible(false);
		} 
	}
	
	/**
	 * Called as a result of implementing KeyListener
	 *
	 * @param e object containing details of the key event
	 */
	public void keyReleased(KeyEvent e)
	{
		// Not used
	}
	
	/**
	 * Called as a result of implementing KeyListener
	 *
	 * @param e object containing details of the key event
	 */
	public void keyTyped(KeyEvent e)
	{
		// Not used
	}
}