/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA
 */
 
package scripting;

import java.io.*;
import FESI.jslib.*;

/**
 * JavaScript / EcmaScript implementation of a language interpreter. This class
 * uses the FESI library by Jean-Marc Lugrin for interpretation, available at:
 * <p><a href="http://www.lugrin.ch/fesi/">http://www.lugrin.ch/fesi/</a><p>
 */
public class JSInterpreterDriver implements InterpreterDriver, Runnable
{
	private static JSGlobalObject interpreter;
	private static JSInterpreterDriver instance;
	
	// Items used by the run() method to interpret in a new thread
	private String scriptData;
	private String argNames[];
	private String argValues[];
	
	static
	{
		try
		{
			instance = new JSInterpreterDriver();
			InterpreterDriverManager.registerDriver(instance);
		}
		catch(Exception e)
		{
			//
		}
	}
	
	/**
	 * Default constructor
	 */
	public JSInterpreterDriver() throws JSException
	{
		// Note: extensions used here in context of language capabilities,
		// NOT file extensions
		String []extensions = new String[] {"FESI.Extensions.BasicIO",
											"FESI.Extensions.FileIO",
											"FESI.Extensions.JavaAccess"};
											
		try {interpreter = JSUtil.makeEvaluator(extensions); }
		catch(JSException e)
		{
			sarj.Sarj.printError("Exception: Cannot initialise FESI interpreter (" + e.getMessage() + ")");
			throw e;
		}
	}
	
	/**
	 * Executes the specified JavaScript/EcmaScript code
	 *
	 * @param script the script data to execute
	 * @param argNames an array containing the names of arguments to be passed to
	 *					   the evaluation
	 * @param argValues an array containing the values of the arguments to be
	 *					  passed to the evaluation
	 *
	 * @throws InterpreterDriver.InterpreterException if an exception occurs during
	 *		   execution
	 */
	public void executeScript(String script, String argNames[], String argValues[]) throws InterpreterDriver.InterpreterException
	{
		scriptData = script;
		this.argNames = argNames;
		this.argValues = argValues;
		
		// The evaluation is executed in a new thread to stop the program
		// from freezing
		Thread evalThread = new Thread(this);
		evalThread.start();
	}
	
	/**
	 * Executes JavaScript/Ecma code contained in the specified file
	 *
	 * @param scriptFile filename containing the code to be executed. This must
	 *		  end in either a '.js' or '.es' extension.
	 * @param argNames an array containing the names of arguments to be passed to
	 *					   the evaluation
	 * @param argValues an array containing the values of the arguments to be
	 *					  passed to the evaluation
	 *
	 * @throws InterpreterDriver.InterpreterException if an exception occurs during
	 *		   execution
	 */
	public void executeScriptFile(String scriptFile, String []argNames, String argValues[]) throws InterpreterDriver.InterpreterException
	{
		try
		{
			scriptData = "";
			String data = new String();
			BufferedReader scriptReader = new BufferedReader(new FileReader(scriptFile));
			while (data != null)
			{
				data = scriptReader.readLine();
				if(data != null)
					scriptData = scriptData + data + "\n";
			}
		}
		catch(FileNotFoundException ex)
		{
			throw new InterpreterDriver.InterpreterException(ex);
		}
		catch(IOException ioe)
		{
			throw new InterpreterDriver.InterpreterException(ioe);
		}
		
		this.argNames = argNames;
		this.argValues = argValues;
		
		// The evaluation is executed in a new thread to stop the program
		// from freezing
		Thread evalThread = new Thread(this);
		evalThread.start();
	}
	
	/**
	 * Returns the list of file extensions supported by this interpreter
	 *
	 * @return an array of strings containing the extensions 'js' and 'es'
	 */
	public String[] getSupportedExtensions()
	{
		return new String[] {"js", "es"};
	}
	
	/**
	 * Returns the list of languages supported by this interpreter
	 *
	 * @return an array of string containing the languages 'JavaScript' and
	 *		   'EcmaScript'
	 */
	public String[] getSupportedLanguages()
	{
		return new String[] {"JavaScript", "EcmaScript"};
	}
	
	/**
	 * Sets a variable/member to the specified value
	 *
	 * @param name the name of the variable to set
	 * @param value the value to assign the variable
	 */
	 public void setVariable(String name, String value)
	 {
	 	if(interpreter != null)
	 	{
	 		try { interpreter.setMember(name, value); }
	 		catch(JSException jse) { }
	 	}
	 }
	 
	 /**
	  * Interprets a script in a new thread
	  */
	 public void run()
	 {
	 	Object result = null;
	 	try
	 	{
	 		
	 		result = interpreter.evalAsFunction(scriptData, argNames, argValues);
	 		
	 		if(result != null)
	 			sarj.Sarj.printError((String)result);
	 	}
	 	catch(JSException ex)
		{
			sarj.Sarj.printError("Interpreter exception: " + ex);
		}
	 }
}