/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */
 
package scripting;

import java.util.*;

/**
 * Class for managing interpreter drivers.
 * This manager is responsible for keeping track of loaded
 * drivers. Interpreter drivers are required to register an
 * instance of themselves with this manager when they are loaded.
 */
public class InterpreterDriverManager
{
	private static HashMap<String, InterpreterDriver> extensionDriverMap = new HashMap<String, InterpreterDriver>();
	private static HashMap<String, InterpreterDriver> languageDriverMap = new HashMap<String, InterpreterDriver>();
	private static ArrayList<InterpreterDriver> driverList = new ArrayList<InterpreterDriver>();
	
	/**
     * Private constructor
     * There is no need for instantiating this class as all methods
     * are static. This private constructor is to disallow creating
     * instances of this class.
     */
    private InterpreterDriverManager()
    {
    	//
    }
    
	/**
     * Register a driver.
     * Interpreter drivers call this method when they are loaded.
     * @param driver the driver to be registered
     */
	public static void registerDriver(InterpreterDriver driver)
	{
		if(driver == null) return;
		
		driverList.add(driver);
		
		String []extensions = driver.getSupportedExtensions();
		for(int i=0; i<extensions.length; i++)
			extensionDriverMap.put(extensions[i], driver);
		
		String []languages = driver.getSupportedLanguages();
		for(int i=0; i<languages.length; i++)
			languageDriverMap.put(languages[i], driver);
	}
	
	/**
     * Execute a script string
     * Execute the string supplied according to the langauge specified
     * @param script script to be executed
     * @param language language for interpreting the script string
     * @param argNames an array containing the names of arguments to be passed to
	 *					   the evaluation
	 * @param argValues an array containing the values of the arguments to be
	 *					  passed to the evaluation
     */
	public static void executeScript(String script, String language, String []argNames, String []argValues) throws InterpreterDriver.InterpreterException
	{
		InterpreterDriver driver = (InterpreterDriver)languageDriverMap.get(language);
		if(driver == null)
		{
			sarj.Sarj.printError("No driver installed to handle language: " + language);
			return;
		}
		driver.executeScript(script, argNames, argValues);
	}
	
	/**
     * Exceute a script file.
     * The interpreter driver supporting the language for this file
     * is deduced from file name extension
     * @param scriptFile file name containing script
     * @param argNames an array containing the names of arguments to be passed to
	 *					   the evaluation
	 * @param argValues an array containing the values of the arguments to be
	 *					  passed to the evaluation
     */
	public static void executeScriptFile(String scriptFile, String []argNames, String []argValues) throws InterpreterDriver.InterpreterException
	{
		if(scriptFile == null) return;
		
		String extension = scriptFile.substring(scriptFile.lastIndexOf('.')+1);
		
		if(extension == null || extension.length() < 0)
		{
			sarj.Sarj.printError("Invalid script file extension: " + scriptFile);
		}
		
		InterpreterDriver driver = (InterpreterDriver)extensionDriverMap.get(extension);
		
		if(driver == null)
		{
			sarj.Sarj.printError("No driver installed to handle extension: " + extension);
			return;
		}
		
		driver.executeScriptFile(scriptFile, argNames, argValues);
	}
	
	/**
	 * Sets a variable/member to the specified value for all registered drivers
	 *
	 * @param name the name of the variable to set
	 * @param value the value to assign the variable
	 */
	 public static void setVariable(String name, String value)
	{
		Iterator<InterpreterDriver> it = driverList.iterator();
		InterpreterDriver driver;
		
		while(it.hasNext())
		{
			driver = it.next();
			driver.setVariable(name, value);
		}
	}
}