/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */
 
package scripting;

/**
 * Defines the interface for a driver for a scripting language interpreter
 */
public interface InterpreterDriver
{
	/**
	 * Executes the script code contained within the specified string
	 *
	 * @param script the script code
	 * @param argNames an array containing the names of arguments to be passed to
	 *					   the evaluation
	 * @param argValues an array containing the values of the arguments to be
	 *					  passed to the evaluation
	 * @throws InterpreterDriver.InterpreterException if an exception occurs
	 *		   while interpreting the script
	 */
	public void executeScript(String script, String[] argNames, String[] argValues) throws InterpreterDriver.InterpreterException;
	
	/**
	 * Executes script contained within a file
	 *
	 * @param scriptFile the file containing the script to be interpreted. This
	 *					 must include a suitable extension for the particular
	 *					 language
	 * @param argNames an array containing the names of arguments to be passed to
	 *					   the evaluation
	 * @param argValues an array containing the values of the arguments to be
	 *					  passed to the evaluation
	 * @throws InterpreterDriver.InterpreterException if an exception occurs
	 *		   while interpreting the script file
	 */
	public void executeScriptFile(String scriptFile, String[] argNames, String[] argValues) throws InterpreterDriver.InterpreterException;
	
	/**
	 * Returns a list of all file extensions associated with the interpreter
	 *
	 * @return an array of string objects containing the supported extensions.
	 *		   These are case-sensitive and must <b>not</b> include the prepending '.'
	 *		   character
	 */
	public String[] getSupportedExtensions();
	
	/**
	 * Returns a list of all languages supported by the interpreter
	 *
	 * @return an array of string objects containing the supported languages.
	 *		   These are case-sensitive.
	 */
	public String[] getSupportedLanguages();
	
	/**
	 * An object of this class is thrown when an exception is generated while
	 * the interpreter is executing a script
	 */
	public static class InterpreterException extends Exception
	{
		private static final long serialVersionUID = 5947847346149275958L;
		
		private Exception underlyingException;
		
		public InterpreterException(Exception e)
		{
			underlyingException = e;
		}
		
		public String toString()
		{
			return "Interpreter Exception (Underlying exception: " + underlyingException + ")";
		}
	}
	
	/**
	 * Sets a variable/member to the specified value
	 *
	 * @param name the name of the variable to set
	 * @param value the value to assign the variable
	 */
	 public void setVariable(String name, String value);
}