/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */
 
package sarj;

/**
 * Defines a listener interface for events generated by user interaction with
 * the GUI
 */
public interface GUIListener
{
	/**
	 * Called when the Connect button is clicked on the GUI
	 *
	 * @param address	the address of the Soldat server, in dotted IP format
	 *					(X.X.X.X)
	 * @param port		the port number of the Soldat server
	 * @param password	the remote administration password for the server
	 */ 
	public void onConnect(String address, int port, String password);
	
	/**
	 * Called when the Disconnect button is clicked on the GUI
	 */
	public void onDisconnect();
	
	/**
	 * Called when the Exit button is clicked on the GUI, or the main GUI window
	 * is closed
	 */
	public void onExit();
	
	/**
	 * Called when the Perform button is clicked on the GUI
	 *
	 * @param actionID the ID of the action to be performed
	 */
	public void onAction(int actionID);
	
	/**
	 * Called when the Refresh button is clicked on the GUI, or the refresh
	 * timer triggers
	 */
	public void onRefresh();
	
	/**
	 * Called when a command is sent from the user
	 *
	 * @param text the command string issued by the user
	 */
	public void onCommand(String text);
	
	/**
	 * Called when the user chooses to kick a player from the list of currently
	 * connected players
	 *
	 * @param playerNumber the unique in-game ID number of the player
	 */
	public void onKickPlayer(int playerNumber);
	
	/**
	 * Called when the user chooses to ban a player from the list of currently
	 * connected players
	 *
	 * @param playerNumber the unique in-game ID number of the player
	 */
	public void onBanPlayer(int playerNumber);
	
	/**
	 * Called when the user chooses to add a player to the remote administrators
	 * list
	 *
	 * @param playerIP the IP address of the player
	 */
	public void onAddAdmin(String playerIP);
	
	/**
	 * Called when the user chooses to remove a player from the remote
	 * administrators list
	 *
	 * @param playerIP the IP address of the player
	 */
	public void onRemoveAdmin(String playerIP);
	
	/**
	 * Called when the user chooses to assign a connected player to a particular
	 * team
	 *
	 * @param playerNumber the unique in-game ID number of the player
	 * @param teamNumber the ID of the team to assign the player to, where
	 *					 1 = alpha, 2 = beta, 3 = charlie and 4 = delta
	 */
	public void onSetTeam(int playerNumber, int teamNumber);
	
	/**
	 * Called when a user clicks a ScriptButton
	 *
	 * @param name the name/caption of the button
	 * @param scriptFile the script file associated with the ScriptButton
	 */
	 public void onScriptButtonClick(String name, String scriptFile);
}