/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */
 
package sarj;

import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.table.*;
import java.io.*;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.Calendar;

/**
 * The GUI class contains all the Swing components that comprise the SARJ
 * Graphical User Interface
 */
public class GUI extends WindowAdapter implements ActionListener, MouseListener, KeyListener
{
	////////////////////////////////////////////////////////////////////////////
	// STATIC VARIABLES
	////////////////////////////////////////////////////////////////////////////
	public final static boolean STATE_CONNECTED = true;
	public final static boolean STATE_DISCONNECTED = false;
	////////////////////////////////////////////////////////////////////////////
	
	// Image file locations
	final static String refreshImageLocation = "/images/refresh.gif";
	final static String performImageLocation = "/images/perform.gif";
	final static String connectImageLocation = "/images/connect.gif";
	final static String disconnectImageLocation = "/images/disconnect.gif";
	final static String exitImageLocation = "/images/exit.gif";
	final static String configImageLocation = "/images/config.gif";
	final static String logoImageLocation = "/images/key.gif";
	final static String scriptSettingsImageLocation = "/images/script.gif";
	final static String addSBImageLocation = "/images/addsb.gif";
	final static String aboutImageLocation = "/images/about.gif";

	private static LogDialog logDialog = new LogDialog();
	
	private JFrame mainFrame;
	private ConfigDialog configDialog;
	private ScriptDialog scriptDialog;
	private AboutDialog aboutDialog;
	private JPanel workspacePanel, leftPanel, rightPanel, gameInfoPanel, actionPanel, refreshButtonPanel;
	private ScriptButtonPanel scriptButtonPanel;
	private JButton connectButton, exitButton, configButton, actionButton, refreshButton, aboutButton;
	private JButton addSBButton, scriptSettingsButton;
	private JTable playerTable, teamTable;
	private JTextArea consoleArea;
	private JComboBox commandBox, actionBox;
	private JTextField ipField, portField;
	private JPasswordField passwordField;
	private JToolBar toolBar, scriptButtonBar;
	private JLabel mapLabel, scoreLimitLabel, gameModeLabel, timeLimitLabel, timeLeftLabel; 
	private ImageIcon connectImage, disconnectImage, configImage, exitImage, scriptSettingsImage, addSBImage, aboutImage;
	private Timer refreshTimer;
	private JScrollPane textScroll, scriptButtonScrollPane;
	private JPopupMenu consoleMenu, playerMenu;
	private JMenu adminSubMenu, teamSubMenu;
	private JMenuItem consoleCutItem, consoleCopyItem, consolePasteItem, consoleClearItem, consoleSaveItem, consoleSelectAllItem, consoleShowLogItem;
	private JMenuItem playerKickItem, playerBanItem, playerAdminAddItem, playerAdminRemoveItem, playerTeamAlphaItem, playerTeamBravoItem, playerTeamCharlieItem, playerTeamDeltaItem;
	private Object []tableCurrentSelection;
	private ArrayList<ScriptButton> SBList;
	private boolean autoRefresh;
	private boolean connectionState;
	private int refreshDelay, refreshClock;
		
	private final String []actionCommands = {"Shutdown server", "Send server message", "Add bot", "Kick player", "Ban player", "Ban IP", "Unban IP", "Change map", "Restart match", "Load next map", "Add remote admin", "Remove remote admin", "Kick last player", "Set respawn time", "Set max respawn time", "Set kill/point/cap limit", "Set time limit", "Set max players", "Set game password", "Set admin password", "Set friendly fire", "Set vote percentage"};
	
	private GUIListener listener;
	
	/**
	 * Creates a new GUI object. The exact appearance of the GUI may change
	 * based on the settings within the provided configuration file
	 *
	 * @param configFilePath the path of the SARJ configuration file
	 *						 (e.g. config.txt)
	 * @param listener the object to receive events generated by the GUI
	 */
	public GUI(String configFilePath, String scriptSettingsFilePath, String scriptButtonSettingsFileLocation, GUIListener listener)
	{
		this.listener = listener;
			
		mainFrame = new JFrame("SARJ " + Sarj.VERSION + " :: Disconnected");
		mainFrame.setDefaultCloseOperation(JFrame.DO_NOTHING_ON_CLOSE);
		mainFrame.getContentPane().setLayout(new BorderLayout());
		mainFrame.addWindowListener(this);
		mainFrame.setFocusable(true);
		mainFrame.addKeyListener(this);
		
		configDialog = new ConfigDialog(this, "Configuration", configFilePath);
		scriptDialog = new ScriptDialog(this, "Script settings", scriptSettingsFilePath);
		aboutDialog = new AboutDialog(mainFrame);
		
		////////////////////////////////////////////////////////////////////////
		// Setup the workspace panel (player list, consoleArea, server info)
		workspacePanel = new JPanel();
		workspacePanel.setLayout(new BoxLayout(workspacePanel, BoxLayout.LINE_AXIS));
		
		////////////////////////////////////////////////////////////////////////
		// Setup the left panel (player list, consoleArea)
		leftPanel = new JPanel();
		leftPanel.setLayout(new BoxLayout(leftPanel, BoxLayout.PAGE_AXIS));
		leftPanel.setBorder(BorderFactory.createEmptyBorder(2,2,2,2));
		////////////////////////////////////////////////////////////////////////
		
		////////////////////////////////////////////////////////////////////////
		// Setup the right panel (server info)
		rightPanel = new JPanel();
		rightPanel.setLayout(new BoxLayout(rightPanel, BoxLayout.PAGE_AXIS));
		rightPanel.setBorder(BorderFactory.createEmptyBorder(2,2,2,2));
		rightPanel.setPreferredSize(new Dimension(200, 420));
		rightPanel.setMinimumSize(new Dimension(200, 420));
		rightPanel.setMaximumSize(new Dimension(200, 420));
		refreshButtonPanel = new JPanel();
		refreshButtonPanel.setLayout(new BoxLayout(refreshButtonPanel, BoxLayout.LINE_AXIS));
		refreshButtonPanel.setMaximumSize(new Dimension(100, 30));
		refreshButton = new JButton("Refresh", new ImageIcon(getClass().getResource(refreshImageLocation)));
		refreshButton.setToolTipText("Refresh the server status");
		
		refreshButtonPanel.add(refreshButton);
		refreshButton.addActionListener(this);
		refreshButton.setEnabled(false);
		
		////////////////////////////////////////////////////////////////////////
		// Setup the game info panel
		gameInfoPanel = new JPanel();
		gameInfoPanel.setPreferredSize(new Dimension(200, 160));
		gameInfoPanel.setMaximumSize(new Dimension(200,160));
		gameInfoPanel.setMinimumSize(new Dimension(200,160));
		gameInfoPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder("Game information"), BorderFactory.createEmptyBorder(0,20,0,20)));
		gameInfoPanel.setLayout(new BoxLayout(gameInfoPanel, BoxLayout.PAGE_AXIS));
		mapLabel = new JLabel("<html>Map: <font color = '#0000FF'><i>?</i></font></html>");
		timeLimitLabel = new JLabel("<html>Time limit: <font color = '#0000FF'><i>?</i></font></html>");
		timeLeftLabel = new JLabel("<html>Time left: <font color = '#0000FF'><i>?</i></font></html>");
		scoreLimitLabel = new JLabel("<html>Score limit: <font color = '#0000FF'><i>?</i></font></html>");
		gameModeLabel = new JLabel("<html>Game: <font color = '#0000FF'><i>?</i></font></html>");
		gameInfoPanel.add(mapLabel);
		gameInfoPanel.add(Box.createRigidArea(new Dimension(0,10)));
		gameInfoPanel.add(timeLimitLabel);
		gameInfoPanel.add(Box.createRigidArea(new Dimension(0,10)));
		gameInfoPanel.add(timeLeftLabel);
		gameInfoPanel.add(Box.createRigidArea(new Dimension(0,10)));
		gameInfoPanel.add(gameModeLabel);
		gameInfoPanel.add(Box.createRigidArea(new Dimension(0,10)));
		gameInfoPanel.add(scoreLimitLabel);
		gameInfoPanel.setAlignmentX(Component.CENTER_ALIGNMENT);
		
		////////////////////////////////////////////////////////////////////////
		// Setup the perform action panel
		actionPanel = new JPanel();
		actionPanel.setLayout(new BoxLayout(actionPanel, BoxLayout.PAGE_AXIS));
		actionPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder("Perform action"), BorderFactory.createEmptyBorder(10,10,10,10)));;
		actionBox = new JComboBox(actionCommands);
		actionBox.setPreferredSize(new Dimension(180, 20));
		actionBox.setFont(new Font("Arial", Font.PLAIN, 12));
		actionBox.setMaximumSize(new Dimension(180, 20));
		actionBox.setEnabled(false);
		actionButton = new JButton("Perform...", new ImageIcon(getClass().getResource(performImageLocation)));
		actionButton.setToolTipText("Perform the selected action");
		actionButton.setAlignmentX(Component.CENTER_ALIGNMENT);
		actionButton.addActionListener(this);
		actionButton.setEnabled(false);
		actionPanel.add(actionBox);
		actionPanel.add(Box.createRigidArea(new Dimension(0,10)));
		actionPanel.add(actionButton);
		
		////////////////////////////////////////////////////////////////////////
		// Setup the toolbar (exit, help, IP, port, connect, etc)
		toolBar = new JToolBar();
		toolBar.setLayout(new FlowLayout(FlowLayout.LEFT, 10, 0));
		connectImage = new ImageIcon(getClass().getResource(connectImageLocation));
		disconnectImage = new ImageIcon(getClass().getResource(disconnectImageLocation));
		connectButton = new JButton(connectImage);
		connectButton.setToolTipText("Connect to the server");
		connectButton.addActionListener(this);
		exitImage = new ImageIcon(getClass().getResource(exitImageLocation));
		exitButton = new JButton(exitImage);
		exitButton.setToolTipText("Exit");
		exitButton.addActionListener(this);
		configImage = new ImageIcon(getClass().getResource(configImageLocation));
		configButton = new JButton(configImage);
		configButton.setToolTipText("Configuration");
		configButton.addActionListener(this);
		ipField = new JTextField();
		ipField.setToolTipText("IP address of the server");
		ipField.setFont(new Font("Lucida Console", Font.PLAIN, 12));
		ipField.setPreferredSize(new Dimension(120, 20));
		ipField.addKeyListener(this);
		portField = new JTextField();
		portField.setToolTipText("Server port");
		portField.setFont(new Font("Lucida Console", Font.PLAIN, 12));
		portField.setPreferredSize(new Dimension(50, 20));
		portField.addKeyListener(this);
		passwordField = new JPasswordField();
		passwordField.setToolTipText("Server password");
		passwordField.setFont(new Font("Lucida Console", Font.PLAIN, 12));
		passwordField.setPreferredSize(new Dimension(100, 20));
		passwordField.addKeyListener(this);
		
		aboutImage = new ImageIcon(getClass().getResource(aboutImageLocation));
		aboutButton = new JButton(aboutImage);
		aboutButton.setToolTipText("About SARJ");
		aboutButton.addActionListener(this);
		
		toolBar.add(connectButton);
		toolBar.add(configButton);
		toolBar.add(exitButton);
		toolBar.add(new JLabel("IP:"));
		toolBar.add(ipField);
		toolBar.add(new JLabel("Port:"));
		toolBar.add(portField);
		toolBar.add(new JLabel("Password:"));
		toolBar.add(passwordField);
		toolBar.addSeparator();
		toolBar.add(aboutButton);
		toolBar.setMinimumSize(new Dimension(620, 20));
		
		scriptButtonBar = new JToolBar();
		scriptButtonBar.setBorderPainted(true);
		
		addSBImage = new ImageIcon(getClass().getResource(addSBImageLocation));
		addSBButton = new JButton("Add script button", addSBImage);
		addSBButton.setToolTipText("Click to add a new ScriptButton");
		addSBButton.addActionListener(this);
		
		scriptSettingsImage = new ImageIcon(getClass().getResource(scriptSettingsImageLocation));
		scriptSettingsButton = new JButton("Script settings", scriptSettingsImage);
		scriptSettingsButton.setToolTipText("View scripting settings");
		scriptSettingsButton.addActionListener(this);
		
		scriptButtonBar.add(scriptSettingsButton);
		scriptButtonBar.add(addSBButton);
		scriptButtonBar.addSeparator();
		
		scriptButtonPanel = new ScriptButtonPanel(mainFrame, scriptButtonSettingsFileLocation, listener);
		scriptButtonScrollPane = new JScrollPane(scriptButtonPanel, ScrollPaneConstants.VERTICAL_SCROLLBAR_NEVER, ScrollPaneConstants.HORIZONTAL_SCROLLBAR_ALWAYS);
		
		scriptButtonBar.add(scriptButtonScrollPane);
		
		
		// Add ScriptButtons to toolbar
		ScriptButton sbs[] = getScriptButtons();
		for(int i=0; i<sbs.length; i++)
			scriptButtonPanel.add(sbs[i]);
			
		scriptButtonScrollPane.setPreferredSize(new Dimension(410, 55));
			
		////////////////////////////////////////////////////////////////////////
		// Setup the player info playerTable
		playerTable = new JTable(new PlayerTableModel());
		playerTable.setToolTipText("Player details");
		playerTable.setPreferredScrollableViewportSize(new Dimension(460, playerTable.getRowHeight() * 4));
		playerTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		playerTable.setForeground(Color.BLUE);
		playerTable.addMouseListener(this);
		playerTable.addKeyListener(this);
		TableColumn tc;
		
		for(int i=0; i<8; i++)
		{
			tc = playerTable.getColumnModel().getColumn(i);
			switch(i)
			{
				case 0:	// Player name
					tc.setPreferredWidth(140);
					break;
				case 1:	// Score
					tc.setPreferredWidth(40);
					break;
				case 2:	// Deaths
					tc.setPreferredWidth(45);
					break;
				case 3:	// Score/Death Ratio
					tc.setPreferredWidth(35);
					break;
				case 4:	// Ping
					tc.setPreferredWidth(35);
					break;
				case 5:	// Team
					tc.setPreferredWidth(45);
					break;
				case 6:	// IP
					tc.setPreferredWidth(100);
					break;
				case 7:	// Number
					tc.setPreferredWidth(20);
					break;
				default:
					break;
			}
		}
		
		JScrollPane playerTableScrollPane = new JScrollPane(playerTable);
		leftPanel.add(playerTableScrollPane);
		leftPanel.add(Box.createRigidArea(new Dimension(0,10)));
		////////////////////////////////////////////////////////////////////////
		
		////////////////////////////////////////////////////////////////////////
		// Setup the teams table
		teamTable = new JTable(new TeamTableModel());
		teamTable.setToolTipText("Team details");
		teamTable.setPreferredScrollableViewportSize(new Dimension(200, teamTable.getRowHeight() * 4));
		teamTable.setAutoResizeMode(JTable.AUTO_RESIZE_OFF);
		teamTable.setForeground(Color.BLUE);
		teamTable.addMouseListener(this);
		
		TableColumn ttc;
		
		for(int i=0; i<5; i++)
		{
			ttc = teamTable.getColumnModel().getColumn(i);
			switch(i)
			{
				case 0:	// Team name
					ttc.setPreferredWidth(45);
					break;
				case 1:	// Score
					ttc.setPreferredWidth(38);
					break;
				case 2:	// Kills
					ttc.setPreferredWidth(30);
					break;
				case 3:	// Deaths
					ttc.setPreferredWidth(45);
					break;
				case 4:	// Ping
					ttc.setPreferredWidth(35);
					break;
				default:
					break;
			}
		}
		
		JScrollPane teamTableScrollPane = new JScrollPane(teamTable);
		teamTableScrollPane.setMaximumSize(new Dimension(Short.MAX_VALUE, 84));
		
		////////////////////////////////////////////////////////////////////////
		// Setup the text area
		consoleArea = new JTextArea(8,6);
		consoleArea.setFont(new Font("Lucida console", Font.PLAIN, 12));
		consoleArea.setForeground(Color.BLUE);
		consoleArea.setBorder(BorderFactory.createEmptyBorder(1,1,1,1));
		consoleArea.addMouseListener(this);
		consoleArea.addKeyListener(this);
		textScroll = new JScrollPane(consoleArea);
		textScroll.setPreferredSize(new Dimension(400, 100));
		leftPanel.add(textScroll);
		////////////////////////////////////////////////////////////////////////
		
		////////////////////////////////////////////////////////////////////////
		// Setup the combo box
		commandBox = new JComboBox();
		commandBox.setEditable(true);
		commandBox.setMaximumSize(new Dimension(Short.MAX_VALUE, 20));
		commandBox.setFont(new Font("Lucida console", Font.PLAIN, 12));
		commandBox.addActionListener(this);
		commandBox.addKeyListener(this);
		commandBox.setEnabled(false);
		leftPanel.add(commandBox);
		////////////////////////////////////////////////////////////////////////
		
		////////////////////////////////////////////////////////////////////////
		// Setup the popup menus
		//
		// Console popup menu
		consoleMenu = new JPopupMenu("Console menu");
		consoleCutItem = new JMenuItem("Cut");
		consoleCutItem.addActionListener(this);
		consoleCopyItem = new JMenuItem("Copy");
		consoleCopyItem.addActionListener(this);
		consolePasteItem = new JMenuItem("Paste");
		consolePasteItem.addActionListener(this);
		consoleClearItem = new JMenuItem("Clear");
		consoleClearItem.addActionListener(this);
		consoleSaveItem = new JMenuItem("Save to File...");
		consoleSaveItem.addActionListener(this);
		consoleSelectAllItem = new JMenuItem("Select All");
		consoleSelectAllItem.addActionListener(this);
		consoleShowLogItem = new JMenuItem("Show/hide log console");
		consoleShowLogItem.addActionListener(this);
		consoleMenu.add(consoleCutItem);
		consoleMenu.add(consoleCopyItem);
		consoleMenu.add(consolePasteItem);
		consoleMenu.addSeparator();
		consoleMenu.add(consoleClearItem);
		consoleMenu.add(consoleSaveItem);
		consoleMenu.addSeparator();
		consoleMenu.add(consoleSelectAllItem);
		consoleMenu.add(consoleShowLogItem);
		
		// Player popup menu
		playerMenu = new JPopupMenu("Player menu");
		adminSubMenu = new JMenu("Remote Admin");
		teamSubMenu = new JMenu("Set Team");
		playerKickItem = new JMenuItem("Kick");
		playerKickItem.addActionListener(this);
		playerBanItem = new JMenuItem("Ban");
		playerBanItem.addActionListener(this);
		playerAdminAddItem = new JMenuItem("Add");
		playerAdminAddItem.addActionListener(this);
		playerAdminRemoveItem = new JMenuItem("Remove");
		playerAdminRemoveItem.addActionListener(this);
		playerTeamAlphaItem = new JMenuItem("Alpha");
		playerTeamAlphaItem.addActionListener(this);
		playerTeamBravoItem = new JMenuItem("Bravo");
		playerTeamBravoItem.addActionListener(this);
		playerTeamCharlieItem = new JMenuItem("Charlie");
		playerTeamCharlieItem.addActionListener(this);
		playerTeamDeltaItem = new JMenuItem("Delta");
		playerTeamDeltaItem.addActionListener(this);
		adminSubMenu.add(playerAdminAddItem);
		adminSubMenu.add(playerAdminRemoveItem);
		teamSubMenu.add(playerTeamAlphaItem);
		teamSubMenu.add(playerTeamBravoItem);
		teamSubMenu.add(playerTeamCharlieItem);
		teamSubMenu.add(playerTeamDeltaItem);
		playerMenu.add(playerKickItem);
		playerMenu.add(playerBanItem);
		playerMenu.add(adminSubMenu);
		playerMenu.add(teamSubMenu);
		//
			
		////////////////////////////////////////////////////////////////////////
		// Add the components
		rightPanel.add(refreshButtonPanel);
		rightPanel.add(Box.createRigidArea(new Dimension(0, 10)));
		//rightPanel.add(scoresPanel);
		rightPanel.add(teamTableScrollPane);
		rightPanel.add(gameInfoPanel);
		rightPanel.add(actionPanel);
		workspacePanel.add(leftPanel);
		workspacePanel.add(rightPanel);
		mainFrame.getContentPane().add(toolBar, BorderLayout.PAGE_START);
		mainFrame.getContentPane().add(scriptButtonBar, BorderLayout.PAGE_END);
		mainFrame.getContentPane().add(workspacePanel, BorderLayout.CENTER);
		mainFrame.getRootPane().setDefaultButton(connectButton);
		mainFrame.setIconImage((new ImageIcon(getClass().getResource(logoImageLocation))).getImage());
		
		configDialog.loadSettings();
		scriptDialog.loadSettings();
		connectionState = STATE_DISCONNECTED;
		
		mainFrame.pack();
		mainFrame.setLocationRelativeTo(null);
	}
	
	/**
	 * Display the GUI
	 */
	public void show()
	{
		mainFrame.setVisible(true);
	}
	
	/**
	 * Hide the GUI
	 */
	public void hide()
	{
		mainFrame.setVisible(false);
		stopRefreshTimer();
	}
	
	/**
	 * Displays error messages in the log console
	 *
	 * @param errorString the error message to display
	 */
	public static void printError(String errorString)
	{
		logDialog.printError(errorString);
	}
	
	/**
	 * Called as a result of implementing ActionListener
	 *
	 * @param e the object containing details of the event
	 */
	public void actionPerformed(ActionEvent e)
	{
		Object source = e.getSource();
		
		if(source == connectButton)
		{
			if(connectionState == STATE_DISCONNECTED)
			{
				String address = getIPField();
				String password = getPasswordField();
				int port;
				
				try{ port = Integer.parseInt(getPortField()); }
				catch(NumberFormatException nfe)
				{
					JOptionPane.showMessageDialog(mainFrame, "Invalid port specified", "Error", JOptionPane.ERROR_MESSAGE);
					return;
				}
				
				int numDots=0;
				for(int i=0; i<address.length(); i++)
				if(address.charAt(i) == '.') numDots++;
				if(numDots != 3)
				{
					JOptionPane.showMessageDialog(mainFrame, "Invalid host IP specified", "Error", JOptionPane.ERROR_MESSAGE);
					return;
				}
				if(port < 1 || port > 65535)
				{
					JOptionPane.showMessageDialog(mainFrame, "Invalid port specified", "Error", JOptionPane.ERROR_MESSAGE);
					return;
				}
				if(password.length() < 1)
				{
					JOptionPane.showMessageDialog(mainFrame, "Please specify a password", "Error", JOptionPane.ERROR_MESSAGE);
					return;
				}
		
				listener.onConnect(address, port, password);
			}
			else
			{
				listener.onDisconnect();
			}
		}
		else if(source == configButton)
		{
			if(!configDialog.isVisible())
				configDialog.setVisible(true);
		}
		else if(source == exitButton)
		{
			listener.onExit();
		}
		else if(source == actionButton)
		{
			listener.onAction(actionBox.getSelectedIndex());
		}
		else if(source == commandBox)
		{
			if("comboBoxEdited".equals(e.getActionCommand()))
			{
				// Send the command
				String commandString = (String)commandBox.getSelectedItem() + "\n";
				commandBox.insertItemAt(commandString, 0);
				commandBox.setSelectedItem("");
				
				listener.onCommand(commandString);
			}
		}
		else if(source == refreshButton)
		{
			listener.onRefresh();
		}
		else if(source == refreshTimer)
		{
			// Count-down the time left field
			if(!timeLeftLabel.getText().contains("?"))
			{
				try
				{
					int minsLeft = Integer.parseInt(timeLeftLabel.getText().substring(44, 46));
					int secsLeft = Integer.parseInt(timeLeftLabel.getText().substring(47,49));
					int timeLeft = minsLeft * 60 + secsLeft;
					timeLeft--;
					
					setTimeLeft("<html>Time left: <font color = '#0000FF'><b>" + Sarj.formatTime(timeLeft) + "</b></font></html>");
				}
				catch(NumberFormatException nfe) { }
			}
			
			if(autoRefresh)
			{
				refreshClock++;
				if(refreshClock >= refreshDelay)
				{
					refreshClock = 0;
					listener.onRefresh();
				}
			}
		}
		else if(source == consoleCutItem)
		{
			consoleArea.cut();
		}
		else if(source == consoleCopyItem)
		{
			consoleArea.copy();
		}
		else if(source == consolePasteItem)
		{
			consoleArea.paste();
		}
		else if(source == consoleClearItem)
		{
			consoleArea.setText("");
		}
		else if(source == consoleSaveItem)
		{
			FileDialog fileDialog = new FileDialog(mainFrame, "Save console to file", FileDialog.SAVE);
			fileDialog.setVisible(true);	
			String fileDir, fileName, filePath;
			fileDir = fileDialog.getDirectory();
			fileName = fileDialog.getFile();
			
			if(fileDir != null && fileName != null)
			{
				filePath = fileDir + fileName;
				BufferedWriter bw;
				
				try
				{
					bw = new BufferedWriter(new FileWriter(new File(filePath)));
					bw.write(consoleArea.getText());
					bw.close();
				}
				catch(IOException ioe)
				{
					JOptionPane.showMessageDialog(mainFrame, "Failed to save to '" + fileName + "'", "Error", JOptionPane.ERROR_MESSAGE);
				}
			}
		}
		else if(source == consoleSelectAllItem)
		{
			consoleArea.selectAll();
		}
		else if(source == consoleShowLogItem)
		{
			logDialog.setVisible(!logDialog.isVisible());
		}
		else if(source == playerKickItem)
		{
			Integer playerNumber = (Integer)tableCurrentSelection[7];
			listener.onKickPlayer(playerNumber.intValue());
		}
		else if(source == playerBanItem)
		{
			Integer playerNumber = (Integer)tableCurrentSelection[7];
			listener.onBanPlayer(playerNumber.intValue());
		}
		else if(source == playerAdminAddItem)
		{
			String playerIP = (String)tableCurrentSelection[6];
			listener.onAddAdmin(playerIP);
		}
		else if(source == playerAdminRemoveItem)
		{
			String playerIP = (String)tableCurrentSelection[6];
			listener.onRemoveAdmin(playerIP);
		}
		else if(source == playerTeamAlphaItem)
		{
			int playerID = ((Integer)tableCurrentSelection[7]).intValue();
			listener.onSetTeam(playerID, 1);
		}
		else if(source == playerTeamBravoItem)
		{
			int playerID = ((Integer)tableCurrentSelection[7]).intValue();
			listener.onSetTeam(playerID, 2);
		}
		else if(source == playerTeamCharlieItem)
		{
			int playerID = ((Integer)tableCurrentSelection[7]).intValue();
			listener.onSetTeam(playerID, 3);
		}
		else if(source == playerTeamDeltaItem)
		{
			int playerID = ((Integer)tableCurrentSelection[7]).intValue();
			listener.onSetTeam(playerID, 4);
		}
		else if(source == scriptSettingsButton)
		{
			scriptDialog.setVisible(true);
		}
		else if (source == addSBButton)
		{
			scriptButtonPanel.addNewButton();
		}
		else if (source == aboutButton)
		{
			aboutDialog.setVisible(true);
		}
	}
	
	/**
	 * Called as a result of implementing KeyListener
	 *
	 * @param e object containing details of the key event
	 */
	public void keyPressed(KeyEvent e)
	{
		if((e.getKeyCode() == KeyEvent.VK_S) && (e.isAltDown()))
		{
			logDialog.setVisible(!logDialog.isVisible());
		} 
	}
	
	/**
	 * Called as a result of implementing KeyListener
	 *
	 * @param e object containing details of the key event
	 */
	public void keyReleased(KeyEvent e)
	{
		// Not used
	}
	
	/**
	 * Called as a result of implementing KeyListener
	 *
	 * @param e object containing details of the key event
	 */
	public void keyTyped(KeyEvent e)
	{
		// Not used
	}
	
	/**
	 * Shows the popup menu if the generated trigger is a popup event for the
	 * current look and feel
	 *
	 * @param e the object containing details of the mouse event
	 */
	private void maybeShowPopup(MouseEvent e)
	{
		if(e.isPopupTrigger())
		{
			if(e.getSource() == consoleArea)
			{
				// See if the cut/copy should be grayed out
				if(consoleArea.getSelectedText() == null)
				{
					consoleCutItem.setEnabled(false);
					consoleCopyItem.setEnabled(false);
				}
				else
				{
					consoleCutItem.setEnabled(true);
					consoleCopyItem.setEnabled(true);
				}
				
				consoleMenu.show(e.getComponent(), e.getX(), e.getY());
			}
			else if(e.getSource() == playerTable)
			{
				int row = playerTable.rowAtPoint(e.getPoint());
				if(row != -1)	// A row is selected
				{
					playerTable.changeSelection(row, 0, false, false);
					tableCurrentSelection = ((PlayerTableModel)playerTable.getModel()).getRow(row);
					teamSubMenu.setEnabled(true);
					playerTeamAlphaItem.setEnabled(true);
					playerTeamBravoItem.setEnabled(true);
					playerTeamCharlieItem.setEnabled(true);
					playerTeamDeltaItem.setEnabled(true);
					// Need to gray out current team
					String team = (String)tableCurrentSelection[5];
						if(team.contains("Alpha"))
							playerTeamAlphaItem.setEnabled(false);
						else if(team.contains("Bravo"))
							playerTeamBravoItem.setEnabled(false);
						else if(team.contains("Charlie"))
							playerTeamCharlieItem.setEnabled(false);
						else if(team.contains("Delta"))
							playerTeamDeltaItem.setEnabled(false);
						else if(team.contains("Spectator"))
							teamSubMenu.setEnabled(false);
					
					playerMenu.show(e.getComponent(), e.getX(), e.getY());	
				}
			}
		}
	}
	
	/**
	 * Sets the delay of the refresh timer for refreshing the player details
	 *
	 * @param delay the time to delay, in milliseconds
	 */
	public void setRefreshDelay(int delay)
	{
		refreshDelay = delay;
	}
	
	public void mouseClicked(MouseEvent e)
	{
		//
	}
	
	public void mouseEntered(MouseEvent e)
	{
		//
	}
	
	public void mouseExited(MouseEvent e)
	{
		//
	}
	
	public void mousePressed(MouseEvent e)
	{
			maybeShowPopup(e);
	}
	
	public void mouseReleased(MouseEvent e)
	{
			maybeShowPopup(e);
	}
	
	/**
	 * Sets the window title of the GUI
	 *
	 * @param title the name of the new title for the window
	 */
	public void setTitle(String title)
	{
		if(title != null)
			mainFrame.setTitle(title);
		else
			mainFrame.setTitle("<null>");
	}
	
	/**
	 * Sets the server address text field on the main window to the one specified
	 *
	 * @param ip the server address to set the text field to
	 */
	public void setIPField(String ip)
	{
		if(ip != null)
			ipField.setText(ip);
		else
			ipField.setText("<null>");
	}
	
	/**
	 * Sets the port number text field on the main window to the one specified
	 *
	 * @param port the port number to set the text field to
	 */
	public void setPortField(String port)
	{
		if(port != null)
			portField.setText(port);
		else
			portField.setText("<null>");
	}
	
	/**
	 * Sets the password field on the main window to the specified string
	 *
	 * @param password the password to set
	 */
	public void setPasswordField(String password)
	{
		if(password != null)
			passwordField.setText(password);
	}
	
	/**
	 * Sets the command combo box field on the main window to the specified string
	 *
	 * @param command the command to set the combo box to
	 */
	public void setCommandBox(String command)
	{
		if(command != null)
			commandBox.setSelectedItem(command);
	}
	
	/**
	 * Returns the value of the command combo box field on the main window
	 *
	 * @return the value of the command combo box
	 */
	public String getCommandBox()
	{
		return (String)commandBox.getSelectedItem();
	}
	
	/**
	 * Sets the automatic refresh timer to on or off
	 *
	 * @param refresh true or false to set the timer to on or off, respectively
	 */
	public void setAutoRefresh(boolean refresh)
	{
		autoRefresh = refresh;
	}
	
	/**
	 * Sets the map name label on the main window to the one specified
	 *
	 * @param mapName the name of the map to set the field to
	 */
	public void setMapName(String mapName)
	{
		mapLabel.setText(mapName);
	}
	
	/**
	 * Sets the game type label on the main window to the one specified
	 *
	 * @param gameType the name of the game type to set the field to
	 *				   (e.g. Deathmatch, CTF, etc)
	 */
	public void setGameType(String gameType)
	{
		gameModeLabel.setText(gameType);
	}
	
	/**
	 * Sets the score limit label on the main window to the one specified
	 *
	 * @param scoreLimit the score limit to set the field to
	 */
	public void setScoreLimit(String scoreLimit)
	{
		scoreLimitLabel.setText(scoreLimit);
	}
	
	/**
	 * Sets the time limit label on the main window to the one specified
	 *
	 * @param timeLimit the time limit to set the field to
	 */
	 public void setTimeLimit(String timeLimit)
	 {
	 	timeLimitLabel.setText(timeLimit);
	 }
	 
	 /**
	  * Sets the time left label on the main window to the one specified
	  *
	  * @param timeLeft the time to set the field to
	  */
	  public void setTimeLeft(String timeLeft)
	  {
	  	timeLeftLabel.setText(timeLeft);
	  }
	
	/**
	 * Clears the connection console
	 */
	public void clearConsole()
	{
		consoleArea.setText("");
	}
	
	/**
	 * Clears the table containing the list of currently connected players
	 */
	public void clearPlayerTable()
	{
		PlayerTableModel ptm = (PlayerTableModel)playerTable.getModel();
		ptm.clear();
	}
	
	/**
	 * Clears the table containing details of each team
	 */
	public void clearTeamTable()
	{
		TeamTableModel ttm = (TeamTableModel)teamTable.getModel();
		ttm.clear();
	}
	
	/**
	 * Adds details of a player to the player table. This should be done for
	 * each connected player after clearing the old details from the table with
	 * the clearPlayerTable method
	 *
	 * @param name the name of the player
	 * @param score the player's score
	 * @param deaths the player's number of deaths
	 * @param ratio the player's kill-to-death ratio (Infinity should be
	 *				represented as '*'
	 * @param ping the player's ping
	 * @param team the player's current team number, where 1 = alpha, 2 = beta,
	 *			   3 = charlie and 4 = delta
	 * @param ip the player's IP address
	 * @param num the player's unique in-game ID, as assigned by the server
	 * 
	 * @see #clearPlayerTable()
	 */
	public synchronized void addPlayer(String name, int score, int deaths, String ratio, int ping, int team, String ip, int num)
	{
		PlayerTableModel ptm = (PlayerTableModel)playerTable.getModel();
		ptm.addPlayer(name, score, deaths, ratio, ping, team, ip, num);
	}
	
	/**
	 * Adds details of a team to the team table. This should be done for each
	 * team, when applicable, after clearing the old details from the table with
	 * the clearTeamTable method
	 *
	 * @param name the name of the team (e.g. Alpha, Beta, Charlie, Delta)
	 * @param score the team's total score
	 * @param kills the total number of kills made by players on the team
	 * @param deaths the total number of deaths made by the players on the team
	 * @param ping the average player ping of the team
	 */
	public void addTeam(String name, int score, int kills, int deaths, int ping)
	{
		TeamTableModel ttm = (TeamTableModel)teamTable.getModel();
		ttm.addTeam(name, score, kills, deaths, ping);
	}
	
	/**
	 * Returns the GUI's main window frame
	 *
	 * @return a JFrame object containing the main window frame
	 */
	public JFrame getMainFrame()
	{
		return mainFrame;
	}
	
	/**
	 * Returns the current value stored in the IP address field on the main window
	 *
	 * @return a String object containing the value of the IP address field
	 */
	public String getIPField()
	{
		return ipField.getText();
	}
	
	/**
	 * Returns the current value stored in the port field on the main window
	 *
	 * @return a String object containing the value of the port field
	 */
	public String getPortField()
	{
		return portField.getText();
	}
	
	/**
	 * Returns the current value stored in the password field on the main window
	 *
	 * @return a String object containing the unmasked value of the password field
	 */
	public String getPasswordField()
	{
		return new String(passwordField.getPassword());
	}
	
	/**
	 * Called when the main window is being closed
	 *
	 * @param e the object containing data of the activating window event
	 */
	public void windowClosing(WindowEvent e)
	{
		configDialog.saveSettings();
		listener.onExit();
	}
	
	/**
	 * Writes the specified string of text to the main console
	 *
	 * @param text the string of text to be printed
	 */
	public void writeToConsole(String text)
	{
		consoleArea.append(text);
		// Set scrollbar position to end of window
		consoleArea.setCaretPosition(consoleArea.getDocument().getLength());
	}
	
	/**
	 * Starts the automatic refresh timer with the specified millisecond delay
	 *
	 * @param delay the timer delay, in milliseconds
	 */
	public void startRefreshTimer(int delay)
	{
		refreshTimer = new Timer(delay, this);
		refreshTimer.setDelay(delay);
		refreshTimer.start();
	}
	
	/**
	 * Stops the automatic refresh timer
	 */
	public void stopRefreshTimer()
	{
		if(refreshTimer != null)
			refreshTimer.stop();
			
		refreshTimer = null;
	}
	
	/**
	 * Updates the GUI components to show that the user is connected. This should
	 * be called whenever a connection is successfully established to a server
	 */
	public void connected()
	{
		actionBox.setEnabled(true);
		actionButton.setEnabled(true);
		commandBox.setEnabled(true);
		refreshButton.setEnabled(true);
		connectButton.setIcon(disconnectImage);
		connectButton.setToolTipText("Disconnect from the server");
		mainFrame.setTitle("SARJ " + Sarj.VERSION + " :: Connected");
		ipField.setEnabled(false);
		portField.setEnabled(false);
		passwordField.setEnabled(false);
		startRefreshTimer(1000);
		
		connectionState = STATE_CONNECTED;
	}
	
	/**
	 * Updates the GUI components to show that the user is disconnected. This
	 * should be called whenever a connection to a server is closed
	 */
	public void disconnected()
	{
		writeToConsole("Connection closed\n");
		actionBox.setEnabled(false);
		actionButton.setEnabled(false);
		commandBox.setEnabled(false);
		refreshButton.setEnabled(false);
		connectButton.setIcon(connectImage);
		connectButton.setToolTipText("Connect to the server");
		ipField.setEnabled(true);
		portField.setEnabled(true);
		passwordField.setEnabled(true);
		mainFrame.setTitle("SARJ " + Sarj.VERSION + " :: Disconnected");
		stopRefreshTimer();
		
		connectionState = STATE_DISCONNECTED;
	}
	
	/**
	 * Adds a new button to the ScriptButton panel
	 *
	 * @param name the name of the button. This will also appear as the button
	 *			   caption
	 * @param command the command string associated with the button, to be
	 *				  executed when it is activated
	 */
	public void addScriptButton(String name, String command)
	{
		scriptButtonPanel.addScriptButton(name, command);
	}
	
	/**
	 * Removes a button from the ScriptButton panel
	 *
	 * @param name the name of the button to remove (the caption of the button)
	 */
	public void removeScriptButton(String name)
	{
		scriptButtonPanel.removeScriptButton(name);
	}
	
	/**
	 * Returns the buttons on the ScriptButton panel
	 *
	 * @return an array of the {@link ScriptButton} objects currently on the panel
	 */
	public ScriptButton[] getScriptButtons()
	{
		return scriptButtonPanel.getScriptButtons();
	}
	
	/**
	 * Returns the name of the file to execute for the specified event
	 *
	 * @param eventName the name of the event
	 * @return the file path of the script to execute
	 * @throws EventNotFoundException if the event isn't found
	 */
	public String getFilenameForEvent(String eventName) throws EventNotFoundException
	{
		return scriptDialog.getFilenameForEvent(eventName);
	}
	
	/**
	 * Returns whether or not the specified event is enabled
	 *
	 * @param eventName the name of the event
	 * @return true if the event is enabled, or false other wise
	 * @throws EventNotFoundException if the event isn't found
	 */
	public boolean isEventEnabled(String eventName) throws EventNotFoundException
	{
		return scriptDialog.isEventEnabled(eventName);
	}
	
	/**
	 * Returns the row of the player table with the player name matching that
	 * provided
	 *
	 * @param playerName the name of the player to match
	 * @return the data contained in the table relating to the player
	 */
	public Object []getPlayerData(String playerName)
	{
		PlayerTableModel model = (PlayerTableModel)playerTable.getModel();
		Object []playerData;
		
		int numRows = model.getRowCount();
		
		for(int i=0; i<numRows; i++)
		{
			playerData = model.getRow(i);
			if(((String)playerData[0]).equals(playerName))
				return playerData;
		}
		
		return new Object[]{};
	}
	
	/**
	 * Returns all the data in the player table
	 * @return a 2D array of objects containing the fields of the player table
	 */
	public Object [][]getPlayerData()
	{
		PlayerTableModel model = (PlayerTableModel)playerTable.getModel();
		
		return model.getData();
	}
	
	/**
	 * Returns the row of the team table with the team name matching that
	 * provided
	 *
	 * @param teamName the name of the team to match
	 * @return the data contained in the table relating to the team
	 */
	public Object []getTeamData(String teamName)
	{
		TeamTableModel model = (TeamTableModel)teamTable.getModel();
		Object []teamData;
		
		int numRows = model.getRowCount();
		
		for(int i=0; i<numRows; i++)
		{
			teamData = model.getRow(i);
			if(((String)teamData[0]).equals(teamName))
				return teamData;
		}
		
		return new Object[]{};
	}
	
	/**
	 * Returns all the data in the team table
	 * @return a 2D array of objects containing the fields of the team table
	 */
	public Object [][]getTeamData()
	{
		TeamTableModel model = (TeamTableModel)teamTable.getModel();
		
		return model.getData();
	}
	
	/**
	 * Checks all of the timed scripts to see if any should run
	 *
	 * @param timePassed the time passed, in milliseconds, since the last call
	 *					  to this function
	 * @return an array of {@link String} objects containing the filenames of the
	 *		   scripts to be executed
	 */
	public String[] checkTimedScripts(long timePassed)
	{
		// Don't run scripts while the dialog is showing - may annoy the user
		if(scriptDialog.isVisible()) return (String [])null;
		
		Object [][]scripts = scriptDialog.getTimedScriptDetails();
		ArrayList<String> scriptList = new ArrayList<String>();
		
		if(scripts == (Object[][])null) return (String [])null;
			
		String filename, type;
		boolean enabled;
		int interval, time;
		
		Calendar currentDate = Calendar.getInstance();
		
		long currentTime = currentDate.get(Calendar.HOUR_OF_DAY) * 3600000 +
							currentDate.get(Calendar.MINUTE) * 60000 +
							currentDate.get(Calendar.SECOND) * 1000 +
							currentDate.get(Calendar.MILLISECOND);
		
		for(int j=0; j<scripts.length; j++)
		{
			filename = (String)scripts[j][0];
			type = (String)scripts[j][1];
			enabled = ((Boolean)scripts[j][2]).booleanValue();
			interval = ((Integer)scripts[j][3]).intValue() * 1000; // seconds -> milliseconds
			time = ((Integer)scripts[j][4]).intValue() * 60000;	// minutes -> milliseconds
				
			if(enabled)
			{	
				if("Time".equals(type))
				{
					// Script is based on a specific time. See if the current time
					// matches the required one
					
					long timeDiff = currentTime - time;
					if(timeDiff >= 0 && timeDiff < Sarj.SCRIPT_TIME_INTERVAL)
					{
						scriptList.add(filename);
					}
				}
				else if("Interval".equals(type))
				{
					if(interval > 0)	// avoid divide-by-zero
					{
						// Script is based on an interval. See if the interval has passed
						if(timePassed % interval < Sarj.SCRIPT_TIME_INTERVAL)
						{
							scriptList.add(filename);
						}
					}
				}
				else
				{
					Sarj.printError("Unknown timed script type: " + type);
				}
			}
		}
		
		return (String [])scriptList.toArray(new String[scriptList.size()]);
	}
}