/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */
 
package sarj;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.io.*;

/**
 * A small dialog allowing the user to change the name and filename of a script
 * button
 */
public class EditScriptDialog extends JDialog implements ActionListener
{
	private static final long serialVersionUID = 5947847346149275958L;
	
	public final static String EXTENSION_js = "js";
    public final static String EXTENSION_es = "es";
    
	ScriptButton button = null;
	ScriptButtonPanel owner;
	
	JButton okButton, cancelButton, scriptChooseButton;
	JTextField nameField, filenameField;
	
	/**
	 * Constructor
	 *
	 * @param owner the script dialog owning this dialog
	 */
	public EditScriptDialog(JFrame parent, String title, ScriptButtonPanel owner)
	{
		super(parent, title, true);
		
		setTitle(title);
		this.owner = owner;
		
		okButton = new JButton("Ok");
		okButton.setToolTipText("Click to accept settings");
		okButton.addActionListener(this);
		cancelButton = new JButton("Cancel");
		cancelButton.setToolTipText("Click to cancel any changes made");
		cancelButton.addActionListener(this);
		
		// The button to open the 'open file' dialog
		scriptChooseButton = new JButton("...");
		scriptChooseButton.setToolTipText("Click to select a script file for this button");
		scriptChooseButton.setPreferredSize(new Dimension(25, 20));
		scriptChooseButton.addActionListener(this);
		
		// The field to store the name of the script button
		nameField = new JTextField(10);
		nameField.setToolTipText("The name of the button");
		nameField.setMaximumSize(new Dimension(240, 20));
		nameField.addActionListener(this);
		
		JPanel namePanel = new JPanel();
		namePanel.setLayout(new BoxLayout(namePanel, BoxLayout.LINE_AXIS));
		namePanel.setAlignmentX(Component.LEFT_ALIGNMENT);
		namePanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		namePanel.add(new JLabel("Name: "));
		namePanel.add(nameField);
		
		// The field to select the script file to execute
		filenameField = new JTextField(20);
		filenameField.setToolTipText("The script file that will be executed when the script button is clicked");
		filenameField.setMaximumSize(new Dimension(Integer.MAX_VALUE, 20));
		filenameField.addActionListener(this);
		
		JPanel fileChoosePanel = new JPanel();
		fileChoosePanel.setLayout(new BoxLayout(fileChoosePanel, BoxLayout.LINE_AXIS));
		fileChoosePanel.setAlignmentX(Component.LEFT_ALIGNMENT);
		fileChoosePanel.setBorder(BorderFactory.createEmptyBorder(5,5,5,5));
		fileChoosePanel.add(new JLabel("Filename: "));
		fileChoosePanel.add(filenameField);
		fileChoosePanel.add(Box.createHorizontalStrut(10));
		fileChoosePanel.add(scriptChooseButton);
		
		JPanel okCancelPanel = new JPanel();
		okCancelPanel.add(okButton);
		okCancelPanel.add(cancelButton);
		
		setLayout(new BorderLayout());
		
		add(namePanel, BorderLayout.NORTH);
		add(fileChoosePanel, BorderLayout.CENTER);
		add(okCancelPanel, BorderLayout.SOUTH);
		
		setLocationRelativeTo(parent);
		setResizable(false);
		pack();
	}
	
	/**
	 * Called as a result of implementing ActionListener
	 *
	 * @param e the object containing details of the event
	 */
	public void actionPerformed(ActionEvent e)
	{
		Object source = e.getSource();
		
		if(source == okButton)
		{
			updateAssociatedButton();
			owner.saveSettings();
			setVisible(false);
		}
		else if(source == cancelButton)
		{
			setVisible(false);
		}
		else if(source == scriptChooseButton)
		{
			JFileChooser fileChooser = new JFileChooser();
			fileChooser.setFileFilter(new javax.swing.filechooser.FileFilter()
			{
				public boolean accept(File f)
				{
				    if (f.isDirectory())
				    {
						return true;
				    }
				
				    String extension = getExtension(f);
				    if (extension != null)
				    {
						if (extension.equals(EXTENSION_js) ||
					    extension.equals(EXTENSION_es))
					    {
					        return true;
					    }
					    else
					    {
					    	return false;
						}
    				}
    				return false;
				}
				
				public String getDescription()
				{
					return "JavaScript / EcmaScript files";
				}
			});
			
			int value = fileChooser.showOpenDialog(this);
			
			if(value == JFileChooser.APPROVE_OPTION)
			{
				File theFile = fileChooser.getSelectedFile();
				
				filenameField.setText(theFile.getAbsolutePath());
			}
		}
	}
	
	/**
	 * Associates this dialog with a particular script button. This should be
	 * called before making the dialog visible
	 *
	 * @param button the script button to associate with the dialog
	 */
	public void associateWith(ScriptButton button)
	{
		this.button = button;
		if(button != null)
		{
			nameField.setText(button.getName());
			filenameField.setText(button.getScriptFile());
		}
	}
	
	/**
	 * Updates the properties of the button associated with the dialog
	 */
	private void updateAssociatedButton()
	{
		if(button != null)
		{
			button.setName(nameField.getText());
			button.setScriptFile(filenameField.getText());
			//button.revalidate();
		}
	}
	
	/*
     * Get the extension of a file.
     *
     * @param f the file to get the extension of
     */
    private String getExtension(File f) {
        String ext = null;
        String s = f.getName();
        int i = s.lastIndexOf('.');

        if (i > 0 &&  i < s.length() - 1) {
            ext = s.substring(i+1).toLowerCase();
        }
        return ext;
    }
}