/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */
 
package sarj;

import java.net.*;
import java.io.*;

/**
 * The Connection class manages a connection with a Soldat server
 */
public class Connection
{
	private Socket socket;
	private ConnectionListener listener;
	private InputStream inputStream;
	
	/**
	 * Creates a new Connection object tied to the specified
	 * {@link ConnectionListener}
	 *
	 * @param listener the object to receive events generated by the connection
	 */
	public Connection(ConnectionListener listener)
	{
		this.listener = listener;
		socket = null;
	}
	
	/**
	 * Connects to the server with the specified address, on the specified port
	 *
	 * @param address	the address of the Soldat server, in dotted IP format
	 *					(X.X.X.X)
	 * @param port		the port number to connect to (23073 by default)
	 */
	public void connect(String address, int port) throws UnknownHostException, IOException
	{
		socket = new Socket(address, port);
		inputStream = socket.getInputStream();
		listener.onConnected();
	}
	
	/**
	 * Closes any open connection to a Soldat server
	 */
	public void disconnect()
	{
		if(socket != null && !socket.isClosed())
		{
			try
			{
				socket.close();
				listener.onDisconnected();
			}
			catch(IOException e)
			{
				listener.onError("IO Exception while disconnecting");
			}
		}
	}
	
	/**
	 * Sends data to the connected Soldat server
	 *
	 * @param data the data to send
	 */
	public void sendData(String data)
	{
		if(socket != null && socket.isConnected() && data != null && data.trim().length() > 0)
		{
			try { socket.getOutputStream().write(data.getBytes()); }
			catch(IOException e)
			{
				if(! socket.isClosed())
				{
					disconnect();
					listener.onError("IO Exception while sending data");
				}
			}
		}
		else
		{
			Sarj.printError("Couldn't send: '" + data + "'");
		}
	}
	
	/**
	 * Tests whether or not the connection socket is closed
	 *
	 * @return true if the connection is closed, or false otherwise
	 */
	public boolean isClosed()
	{
		if(socket != null)
			return socket.isClosed();
		else
			return true;
	}
	
	/**
	 * Tests whether or not the connection socket is connected
	 *
	 * @return true if the socket is connected, or false otherwise
	 */
	public boolean isConnected()
	{
		if(socket != null)
			return socket.isConnected();
		else
			return false;
	}
	
	/**
	 * Reads a single character from an open connection. This method blocks
	 * until completed
	 *
	 * @return the character read from the connection socket
	 * @throws IOException if an input or output error occurred
	 */
	public int read() throws IOException
	{
		if(inputStream == null)
			inputStream = socket.getInputStream();
			
		return inputStream.read();
	}
	
	/**
	 * Reads characters up to and including a newline character from the
	 * connection socket
	 *
	 * @return a String object containing the received string
	 * @throws IOException if an input or output error occurred
	 */
	public String readString() throws IOException
	{
		if(inputStream == null)
			inputStream = socket.getInputStream();
	
		StringBuffer sb = new StringBuffer();
		int i=0;
		
		while(i != 0x0A && i !=-1)
		{
			i = inputStream.read();
			if(i==-1)
				throw new IOException();
			else
				sb.append((char)i);
		}
		
		return sb.toString();
	}
	
	/**
	 * Skips the specified number of bytes of data waiting on the connection
	 * socket
	 *
	 * @param num the number of bytes to skip
	 */
	public void skip(int num) throws IOException
	{
		if(inputStream == null)
			inputStream = socket.getInputStream();
			
		inputStream.skip(num);
	}
}