/*
 * Copyright 2005 Richard Davies
 *
 *
 * This file is part of SARJ.
 *
 * SARJ is free software; you can redistribute it and/or modify it under the terms of
 * the GNU General Public License as published by the Free Software Foundation;
 * either version 2 of the License, or (at your option) any later version.
 * 
 * SARJ is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 * PURPOSE. See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along with
 * SARJ; if not, write to the Free Software Foundation, Inc., 59 Temple Place,
 * Suite 330, Boston, MA 02111-1307 USA 
 */

package sarj;

import java.io.*;
import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.border.*;

/**
 * The ConfigDialog class contains the components of the Configuration window
 */
public class ConfigDialog extends JDialog implements ActionListener, ItemListener
{
	////////////////////////////////////////////////////////////////////////////
	// Class variables
	////////////////////////////////////////////////////////////////////////////
	// Instance
	private String configFile;
	private JPanel refreshPanel, refreshTimePanel, buttonPanel, lookAndFeelPanel;
	private JCheckBox autoRefreshBox, javaUIBox;
	private JTextField refreshTimeField;
	private JButton okButton;
	private GUI owner;
	private JLabel preRefreshLabel, postRefreshLabel, noteLabel;
	// Class
	private static boolean javaUI;	// flags whether using Java 'look-and-feel' or not
	private static final long serialVersionUID = 5947847346149275958L;
	
	/**
	 * Creates a new ConfigDialog object
	 *
	 * @param owner the parent GUI
	 * @param title title for the configuration window
	 * @param configFile the path to the configuration file
	 */
	public ConfigDialog(GUI owner, String title, String configFile)
	{
		super(owner.getMainFrame(), title, true);
		this.configFile = configFile;
		this.owner = owner;
		setLocationRelativeTo(null);
		getContentPane().setLayout(new BoxLayout(getContentPane(), BoxLayout.PAGE_AXIS));
		
		refreshPanel = new JPanel();
		refreshPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(5,5,5,5), BorderFactory.createTitledBorder("Refresh options")));
		refreshPanel.setLayout(new BoxLayout(refreshPanel, BoxLayout.PAGE_AXIS));
		
		autoRefreshBox = new JCheckBox("Automatically refresh server status");
		autoRefreshBox.setAlignmentX(Component.CENTER_ALIGNMENT);
		autoRefreshBox.addItemListener(this);
		refreshPanel.add(autoRefreshBox);
		
		refreshTimePanel = new JPanel();
		refreshTimePanel.setLayout(new BoxLayout(refreshTimePanel, BoxLayout.LINE_AXIS));
		preRefreshLabel = new JLabel("Refresh delay:");
		refreshTimePanel.add(preRefreshLabel);
		refreshTimePanel.add(Box.createRigidArea(new Dimension(10,0)));
		refreshTimeField = new JTextField(3);
		refreshTimeField.setPreferredSize(new Dimension(30, 20));
		refreshTimeField.setMinimumSize(new Dimension(30, 20));
		refreshTimeField.setMaximumSize(new Dimension(30, 20));
		refreshTimePanel.add(refreshTimeField);
		refreshTimePanel.add(Box.createRigidArea(new Dimension(10,0)));
		postRefreshLabel = new JLabel("seconds");
		refreshTimePanel.add(postRefreshLabel);
		refreshPanel.add(refreshTimePanel);
	
		buttonPanel = new JPanel();
		okButton = new JButton("Ok", Sarj.OK_ICON);
		okButton.addActionListener(this);
		buttonPanel.add(okButton);
		
		lookAndFeelPanel = new JPanel();
		lookAndFeelPanel.setLayout(new BoxLayout(lookAndFeelPanel, BoxLayout.PAGE_AXIS));
		lookAndFeelPanel.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createTitledBorder("Look and Feel"), BorderFactory.createEmptyBorder(5,5,5,5)));
		javaUIBox = new JCheckBox("Use Java Look and Feel");
		javaUIBox.addActionListener(this);
		lookAndFeelPanel.add(javaUIBox);
		noteLabel = new JLabel("<html><font size='2'>Note: You must restart SARJ for changes to take effect</font></html>");
		noteLabel.setAlignmentX(Component.LEFT_ALIGNMENT);
		lookAndFeelPanel.setAlignmentX(Component.CENTER_ALIGNMENT);
		lookAndFeelPanel.add(noteLabel);
			
		getContentPane().add(refreshPanel);
		getContentPane().add(lookAndFeelPanel);
		getContentPane().add(buttonPanel);
		setResizable(false);
		
		pack();
		setLocationRelativeTo(null);

	}
	
	/**
	 * Called as a result of implementing ActionListener
	 *
	 * @param e the object containing details of the event
	 */
	public void actionPerformed(ActionEvent e)
	{
		if(e.getSource() == okButton)
		{
			saveSettings();
			setVisible(false);
		}
		else if(e.getSource() == javaUIBox)
		{
			if(javaUIBox.isSelected())
				javaUI = true;
			else
				javaUI = false;
		}
	}
	
	/**
	 * Loads settings from the file with the path specified in the configFile field
	 */
	public void loadSettings()
	{
		BufferedReader br;
		String configData = "";
		String []tokens;
		
		// Set everything to default here
		owner.setIPField("0.0.0.0");
		owner.setPortField("23073");
		owner.setAutoRefresh(true);
		autoRefreshBox.setSelected(true);
		preRefreshLabel.setEnabled(true);
		refreshTimeField.setEnabled(true);
		postRefreshLabel.setEnabled(true);
		owner.setRefreshDelay(5);
		
		try
		{
			br = new BufferedReader(new FileReader(new File(configFile)));
		}
		catch(Exception e)
		{
			JOptionPane.showMessageDialog(owner.getMainFrame(), "Failed to open config file! Using default settings.", "Error", JOptionPane.ERROR_MESSAGE);
			return;
		}
		try
		{
			configData = br.readLine();
		}
		catch(IOException ioe)
		{
			JOptionPane.showMessageDialog(owner.getMainFrame(), "Warning: Exception occurred when reading config file! Using default settings.", "Error", JOptionPane.ERROR_MESSAGE);
			return;
		}
		while(configData != null)
		{
			try
			{
				
				if(configData != null && configData.length() > 0 && !configData.trim().startsWith("#"))
				{
					tokens = configData.split("=");
					if(tokens.length > 1)
					{
						// Remove padding spaces
						for(int i=0; i<tokens.length; i++)
							tokens[i] = tokens[i].trim();
							
						// Check which config element was set
						if("AutoRefresh".equalsIgnoreCase(tokens[0]))
						{
							if("1".equals(tokens[1]) || "true".equalsIgnoreCase(tokens[1]) || "yes".equalsIgnoreCase(tokens[1]) || "on".equalsIgnoreCase(tokens[1]))
							{
								// Auto refresh was set to 'true'
								owner.setAutoRefresh(true);
								autoRefreshBox.setSelected(true);
								preRefreshLabel.setEnabled(true);
								refreshTimeField.setEnabled(true);
								postRefreshLabel.setEnabled(true);
							}
							else
							{
								owner.setAutoRefresh(false);
								autoRefreshBox.setSelected(false);
								preRefreshLabel.setEnabled(false);
								refreshTimeField.setEnabled(false);
								postRefreshLabel.setEnabled(false);
							}
						}
						else if("RefreshDelay".equalsIgnoreCase(tokens[0]))
						{
							int i;
							
							try
							{
								i = Integer.parseInt(tokens[1]);
							}
							catch(NumberFormatException nfe)
							{
								// If erroneous set to default of 5
								i = 5;
							}
							// If number was below 1, set to default
							if(i < 1) i = 5;
							refreshTimeField.setText((new Integer(i)).toString());
							owner.setRefreshDelay(i);
						}
						else if("IP".equalsIgnoreCase(tokens[0]))
						{
							owner.setIPField(tokens[1]);
						}
						else if("Port".equalsIgnoreCase(tokens[0]))
						{
							owner.setPortField(tokens[1]);
						}
						else if("JavaUI".equalsIgnoreCase(tokens[0]))
						{
							if("1".equals(tokens[1]) || "true".equalsIgnoreCase(tokens[1]) || "yes".equalsIgnoreCase(tokens[1]) || "on".equalsIgnoreCase(tokens[1]))
							{
								// JavaUI was selected
								javaUI = true;
								javaUIBox.setSelected(true);
							}
							else
							{
								javaUI = false;
								javaUIBox.setSelected(false);
							}
						}
					}
				}
				
				// Read and parse a line
				configData = br.readLine();
			}
			catch(IOException ioee)
			{
				JOptionPane.showMessageDialog(owner.getMainFrame(), "Warning: Failed to read from config file!", "Error", JOptionPane.ERROR_MESSAGE);
				return;
			}
		}
		try
		{
			br.close();
		}
		catch(IOException closeEx)
		{
			JOptionPane.showMessageDialog(owner.getMainFrame(), "Warning: Exception occurred when closing config file!", "Error", JOptionPane.ERROR_MESSAGE);
		}
		
	}
	
	/**
	 * Saves the current settings to the file with the path specified in the
	 * configFile field
	 */
	public void saveSettings()
	{
		BufferedWriter bw;
		boolean autoRefresh;
		int refreshDelay;
		
		try
		{
			refreshDelay = Integer.parseInt(refreshTimeField.getText());
		}
		catch(NumberFormatException nfe)
		{
			refreshDelay = 5;
		}
		if(refreshDelay < 0)
			refreshDelay = 5;	// Set to default
			
		if(autoRefreshBox.isSelected())
			autoRefresh = true;
		else
			autoRefresh = false;
		
		// First apply the settings to the main Sarj object
		owner.setAutoRefresh(autoRefresh);
		owner.setRefreshDelay(refreshDelay);
		
		try
		{
			bw = new BufferedWriter(new FileWriter(new File(configFile)));
			bw.write("# SARJ configuration settings\r\n\r\n");
			bw.write("AutoRefresh=" + (autoRefresh? "1" : "0") + "\r\n");
			bw.write("RefreshDelay=" + refreshDelay + "\r\n");
			bw.write("IP=" + owner.getIPField() + "\r\n");
			bw.write("Port=" + owner.getPortField() + "\r\n");
			bw.write("JavaUI=" + (javaUI ? "1" : "0") + "\r\n");
			bw.close();
		}
		catch(IOException ioe)
		{
			JOptionPane.showMessageDialog(owner.getMainFrame(), "Warning: Failed to save configuration settings!", "Error", JOptionPane.ERROR_MESSAGE);
			return;
		}
	}
	
	/**
	 * Called as a result of implementing ItemListener
	 *
	 * @param e the object containing details of the event
	 */
	public void itemStateChanged(ItemEvent e)
	{
		Object source = e.getItemSelectable();
		
		if(source == autoRefreshBox)
		{
			if(autoRefreshBox.isSelected())
			{
				preRefreshLabel.setEnabled(true);
				refreshTimeField.setEnabled(true);
				postRefreshLabel.setEnabled(true);
			}
			else
			{
				preRefreshLabel.setEnabled(false);
				refreshTimeField.setEnabled(false);
				postRefreshLabel.setEnabled(false);
			}
		}
	}
	
	/**
	 * Returns the name of the UI style (Java/Windows) from the file with
	 * the path specified. This should be called before loading other settings,
	 * as Java requires the look-and-feel to be specified before any GUI
	 * components are created
	 *
	 * @param configFile path of the configuration file
	 */
	static public String getUIClassNameFromConfigFile(String configFile)
	{
		BufferedReader br;
		String tokens[];
		String configData;
		
		try
		{
			br = new BufferedReader(new FileReader(new File(configFile)));
		}
		catch(Exception e)
		{
			javaUI = false;
			return UIManager.getSystemLookAndFeelClassName();
		}
		try
		{
			configData = br.readLine();
		}
		catch(IOException ioe)
		{
			javaUI = false;
			return UIManager.getSystemLookAndFeelClassName();
		}
		while(configData != null)
		{
			try
			{
				if(configData != null && configData.length() > 0 && !configData.trim().startsWith("#"))
				{
					tokens = configData.split("=");
					if(tokens.length > 1)
					{
						// Remove padding spaces
						for(int i=0; i<tokens.length; i++)
							tokens[i] = tokens[i].trim();
							
						// Check which config element was set
						if("JavaUI".equalsIgnoreCase(tokens[0]))
						{
							if("1".equals(tokens[1]) || "true".equalsIgnoreCase(tokens[1]) || "yes".equalsIgnoreCase(tokens[1]) || "on".equalsIgnoreCase(tokens[1]))
							{
								javaUI = true;
								return UIManager.getCrossPlatformLookAndFeelClassName();
							}
							else
							{
								javaUI = false;
								return UIManager.getSystemLookAndFeelClassName();
							}
						}
					}
				}
				configData = br.readLine();
			}
			catch(Exception e)
			{
				javaUI = false;
				return UIManager.getSystemLookAndFeelClassName();
			}
		}
		javaUI = false;
		return UIManager.getSystemLookAndFeelClassName();
	}
}